/**
 * Copyright(c) Foresee Science & Technology Ltd.
 */
package com.foresee.open.sdk.kit.gm;

import com.foresee.open.sdk.kit.ZipUtil;
import sun.misc.BASE64Decoder;
import sun.misc.BASE64Encoder;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.nio.charset.Charset;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>Title: <国密SM4></p>
 * <p>Description: <代码取自网上开源项目，有改动。></p>
 *
 * @author huangzhongying@foresee.com.cn
 * @date created on 2019/9/29 15:05
 */
public class SM4Util {
    private static String charsetName = "UTF8";
    private static Pattern p = Pattern.compile("\\s*|\t|\r|\n");

    public enum SM4CIPHER {
        ECB,
        CBC
    }


    /**
     * 默认加密方式
     */
    private static final SM4CIPHER DEFAULT_CIPHER = SM4CIPHER.CBC;


    public static void main(String[] args) {
        String data = "123456";
        String key = "abc123";

        String encryptData = SM4Util.encryptDataCBC(key, data);
        System.err.println("encryptData:" + encryptData);
        String encryptData2 = SM4Util.encryptDataCBC2(key, data);
        System.err.println("encryptData2:" + encryptData2);
        String decryptData = SM4Util.decryptDataCBC(key, encryptData);
        System.err.println("decryptData:" + decryptData);
        String decryptData2 = SM4Util.decryptDataCBC2(key, encryptData2);
        System.err.println("decryptData2:" + decryptData2);


        String encryptData3 = SM4Util.encryptDataECB(key, data);
        System.err.println("encryptData3:" + encryptData3);
        String encryptData4 = SM4Util.encryptDataECB2(key, data);
        System.err.println("encryptData4:" + encryptData4);
        String decryptData3 = SM4Util.decryptDataECB(key, encryptData3);
        System.err.println("decryptData3:" + decryptData3);
        String decryptData4 = SM4Util.decryptDataECB2(key, encryptData4);
        System.err.println("decryptData4:" + decryptData4);

        String encryptData5 = SM4Util.encryptDataCBC(key, key + "1", data);
        System.err.println("encryptData5:" + encryptData5);
        String encryptData6 = SM4Util.encryptDataCBC2(key, key + "1", data);
        System.err.println("encryptData6:" + encryptData6);
        String decryptData5 = SM4Util.decryptDataCBC(key, key + "1", encryptData5);
        System.err.println("decryptData5:" + decryptData5);
        String decryptData6 = SM4Util.decryptDataCBC2(key, key + "1", encryptData6);
        System.err.println("decryptData6:" + decryptData6);


        String encryptData7 = SM4Util.encryptDataCBC(key, key + "1", data,true);
        System.err.println("encryptData7:" + encryptData7);
        String decryptData7 = SM4Util.decryptDataCBC(key, key + "1", encryptData7,true);
        System.err.println("decryptData7:" + decryptData7);

        String encryptData8 = SM4Util.encryptDataECB(key, data,true);
        System.err.println("decryptData8:" + encryptData8);
        String decryptData8 = SM4Util.decryptDataECB(key, encryptData8,true);
        System.err.println("decryptData8:" + decryptData8);
    }

    /**
     * ECB加密方式
     *
     * @param data      :待加密数据
     * @param secretKey :加密KEY
     * @return
     */
    public static String encryptDataECB(String secretKey, String data) {
        return encryptDataECB(secretKey, data, false);
    }

    /**
     * ECB解密方式
     *
     * @param secretKey :解密KEY
     * @param data      :待解密数据
     * @return
     */
    public static String decryptDataECB(String secretKey, String data) {
        return decryptDataECB(secretKey, data, false);
    }


    /**
     * ECB加密方式
     *
     * @param data      :待加密数据
     * @param secretKey :加密KEY
     * @param zip       :是否压缩
     * @return
     */
    public static String encryptDataECB(String secretKey, String data, boolean zip) {
        if (zip) {
            String zipData=new String(ZipUtil.compress(data.getBytes(Charset.defaultCharset())), Charset.defaultCharset());
            return encryptData(secretKey, null,ZipUtil.compress(data.getBytes(Charset.defaultCharset())), SM4CIPHER.ECB);
        }
        return encryptData(secretKey, null, data.getBytes(), SM4CIPHER.ECB);
    }

    /**
     * ECB解密方式
     *
     * @param secretKey :解密KEY
     * @param data      :待解密数据
     * @param zip       :是否压缩
     * @return
     */
    public static String decryptDataECB(String secretKey, String data, boolean zip) {
        byte[] decryptDatas = decryptData(secretKey, null, data, SM4CIPHER.ECB);
        if (zip) {
            return new String(ZipUtil.decompress(decryptDatas), Charset.defaultCharset());
        }
        return toString(decryptDatas);
    }

    /**
     * CBC加密方式
     *
     * @param data      :待加密数据
     * @param secretKey :加密KEY
     * @return
     */
    public static String encryptDataCBC(String secretKey, String data) {
        return encryptDataCBC(secretKey, secretKey, data, false);
    }

    /**
     * CBC解密方式
     *
     * @param data      :待解密数据
     * @param secretKey :解密KEY
     * @return
     */
    public static String decryptDataCBC(String secretKey, String data) {
        return decryptDataCBC(secretKey, secretKey, data, false);
    }


    /**
     * CBC加密方式
     *
     * @param data      :待加密数据
     * @param secretKey :加密KEY
     * @param iv        :加密KEY2
     * @return
     */
    public static String encryptDataCBC(String secretKey, String iv, String data) {
        return encryptDataCBC(secretKey, iv, data, false);
    }


    /**
     * CBC加密方式
     *
     * @param data      :待加密数据
     * @param secretKey :加密KEY
     * @param iv        :加密KEY2
     * @param zip       :是否压缩
     * @return
     */
    public static String encryptDataCBC(String secretKey, String iv, String data, boolean zip) {
        if (zip) {
            return encryptData(secretKey, iv, ZipUtil.compress(data.getBytes()), SM4CIPHER.CBC);
        }
        return encryptData(secretKey, iv, data.getBytes(), SM4CIPHER.CBC);
    }


    /**
     * CBC解密方式
     *
     * @param data      :待解密数据
     * @param secretKey :解密KEY
     * @param iv        :加密KEY2
     * @return
     */
    public static String decryptDataCBC(String secretKey, String iv, String data) {
        return decryptDataCBC(secretKey, iv, data, false);
    }

    /**
     * CBC解密方式
     *
     * @param data      :待解密数据
     * @param secretKey :解密KEY
     * @param zip       :是否压缩
     * @return
     */
    public static String decryptDataCBC(String secretKey, String data, boolean zip) {
        return decryptDataCBC(secretKey, secretKey, data, zip);
    }

    /**
     * CBC加密方式
     *
     * @param data      :待加密数据
     * @param secretKey :加密KEY
     * @param zip       :是否压缩
     * @return
     */
    public static String encryptDataCBC(String secretKey, String data, boolean zip) {
        return encryptDataCBC(secretKey, secretKey, data, zip);
    }


    /**
     * CBC解密方式
     *
     * @param data      :待解密数据
     * @param secretKey :解密KEY
     * @param iv        :解密KEY2
     * @param zip       :是否压缩
     * @return
     */
    public static String decryptDataCBC(String secretKey, String iv, String data, boolean zip) {
        byte[] decryptDatas= decryptData(secretKey, iv, data, SM4CIPHER.CBC);
        if (zip) {
            return toString(ZipUtil.decompress(decryptDatas));
        }
        return toString(decryptDatas);
    }


    /**
     * 加密数据
     *
     * @param secretKey :加密KEY
     * @param iv        :加密KEY2,ECB模式时，为null,CBC模式，必填。默认等于secretKey
     * @param datas      :待加密的数据
     * @param sm4CIPHER :加密模式
     * @return
     */
    private static String encryptData(String secretKey, String iv, byte[] datas, SM4CIPHER sm4CIPHER) {
        try {
            SM4Context ctx = new SM4Context();
            ctx.isPadding = true;
            ctx.mode = SM4Core.SM4_ENCRYPT;
            byte[] keyBytes = get16CharIfNecessary(secretKey).getBytes();
            SM4Core sm4 = new SM4Core();
            // 设置密钥
            sm4.sm4_setkey_enc(ctx, keyBytes);
            byte[] encrypted = null;
            //加密方式为ECB时
            if (SM4CIPHER.ECB == sm4CIPHER) {
                // 进行加密
                encrypted = sm4.sm4_crypt_ecb(ctx, datas);
            } else if (SM4CIPHER.CBC == sm4CIPHER) {
                byte[] ivBytes = get16CharIfNecessary(iv).getBytes();
                encrypted = sm4.sm4_crypt_cbc(ctx, ivBytes,datas);
            } else {
                return null;
            }
            String cipherText = new BASE64Encoder().encode(encrypted);
            /*if (cipherText != null && cipherText.trim().length() > 0) {
                Matcher m = p.matcher(cipherText);
                cipherText = m.replaceAll("");
            }*/
            return cipherText;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

    }

    /**
     * 解密数据
     *
     * @param secretKey :解密KEY
     * @param iv        :解密KEY2,ECB模式时，为null,CBC模式，必填。默认等于secretKey
     * @param data      :待解密的数据
     * @param sm4CIPHER :解密模式
     * @return
     */
    private static byte[] decryptData(String secretKey, String iv, String data, SM4CIPHER sm4CIPHER) {
        try {
            SM4Context ctx = new SM4Context();
            ctx.isPadding = true;
            ctx.mode = SM4Core.SM4_DECRYPT;
            byte[] keyBytes = get16CharIfNecessary(secretKey).getBytes();
            SM4Core sm4 = new SM4Core();
            sm4.sm4_setkey_dec(ctx, keyBytes);
            byte[] decrypted = null;
            if (SM4CIPHER.ECB == sm4CIPHER) {
                decrypted = sm4.sm4_crypt_ecb(ctx, new BASE64Decoder().decodeBuffer(data));
            } else if (SM4CIPHER.CBC == sm4CIPHER) {
                byte[] ivBytes = get16CharIfNecessary(iv).getBytes();
                decrypted = sm4.sm4_crypt_cbc(ctx, ivBytes, new BASE64Decoder().decodeBuffer(data));
            }
            return decrypted;
        } catch (Exception e) {

        }
        return null;
    }

    private static String toString(byte[] datas){
        try {
            return new String(datas,charsetName);
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    /**
     * ECB迭代（分组）模式加密
     * 加密模式说明参见：https://www.cnblogs.com/starwolf/p/3365834.html
     *
     * @param secretKey 密钥
     * @param plainText 明文
     * @return 密文
     */
    public static String encryptDataECB2(String secretKey, String plainText) {
        try {
            SM4Context ctx = new SM4Context();
            ctx.isPadding = true;
            ctx.mode = SM4Core.SM4_ENCRYPT;

            byte[] keyBytes = get16CharIfNecessary(secretKey).getBytes();

            SM4Core sm4 = new SM4Core();
            // 设置密钥
            sm4.sm4_setkey_enc(ctx, keyBytes);
            // 进行加密
            byte[] encrypted = sm4.sm4_crypt_ecb(ctx, plainText.getBytes(charsetName));

            String cipherText = new BASE64Encoder().encode(encrypted);
            if (cipherText != null && cipherText.trim().length() > 0) {

                Matcher m = p.matcher(cipherText);
                cipherText = m.replaceAll("");
            }
            return cipherText;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static String decryptDataECB2(String secretKey, String cipherText) {
        try {
            SM4Context ctx = new SM4Context();
            ctx.isPadding = true;
            ctx.mode = SM4Core.SM4_DECRYPT;

            byte[] keyBytes = get16CharIfNecessary(secretKey).getBytes();

            SM4Core sm4 = new SM4Core();
            sm4.sm4_setkey_dec(ctx, keyBytes);
            byte[] decrypted = sm4.sm4_crypt_ecb(ctx, new BASE64Decoder().decodeBuffer(cipherText));
            return new String(decrypted, charsetName);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static String encryptDataCBC2(String secretKey, String plainText) {
        return encryptDataCBC2(secretKey, secretKey, plainText);
    }

    /**
     * 密码分组链接模式加密
     * 加密模式说明参见：https://www.cnblogs.com/starwolf/p/3365834.html
     *
     * @param secretKey 密钥
     * @param iv        加密初始向量
     * @param plainText 明文
     * @return 密文
     */
    public static String encryptDataCBC2(String secretKey, String iv, String plainText) {

        try {
            SM4Context ctx = new SM4Context();
            ctx.isPadding = true;
            ctx.mode = SM4Core.SM4_ENCRYPT;

            byte[] keyBytes = get16CharIfNecessary(secretKey).getBytes();
            byte[] ivBytes = get16CharIfNecessary(iv).getBytes();

            SM4Core sm4 = new SM4Core();
            sm4.sm4_setkey_enc(ctx, keyBytes);
            byte[] encrypted = sm4.sm4_crypt_cbc(ctx, ivBytes, plainText.getBytes(charsetName));
            String cipherText = new BASE64Encoder().encode(encrypted);
            if (cipherText != null && cipherText.trim().length() > 0) {
                Matcher m = p.matcher(cipherText);
                cipherText = m.replaceAll("");
            }
            return cipherText;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static String decryptDataCBC2(String secretKey, String cipherText) {
        return decryptDataCBC2(secretKey, secretKey, cipherText);
    }

    public static String decryptDataCBC2(String secretKey, String iv, String cipherText) {
        try {
            SM4Context ctx = new SM4Context();
            ctx.isPadding = true;
            ctx.mode = SM4Core.SM4_DECRYPT;

            byte[] keyBytes = get16CharIfNecessary(secretKey).getBytes();
            byte[] ivBytes = get16CharIfNecessary(iv).getBytes();

            SM4Core sm4 = new SM4Core();
            sm4.sm4_setkey_dec(ctx, keyBytes);
            byte[] decrypted = sm4.sm4_crypt_cbc(ctx, ivBytes, new BASE64Decoder().decodeBuffer(cipherText));
            return new String(decrypted, charsetName);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 返回16位字符串
     * 如果入参长度不是16位，则取hash值后截取16位。
     */
    private static String get16CharIfNecessary(String text) {
        if (text == null) {
            throw new IllegalArgumentException("argument err");
        }
        if (text.length() != 16) {
            return SM3Util.hash(text).substring(0, 16);
        }
        return text;
    }

    /**
     * SM4核心算法类。
     * <p>
     * 详见国家密码局发布的《SM4 Block Cipher Algorithm.pdf》
     */
    private static class SM4Core {
        public static final int SM4_ENCRYPT = 1;

        public static final int SM4_DECRYPT = 0;

        private long GET_ULONG_BE(byte[] b, int i) {
            long n = (long) (b[i] & 0xff) << 24 | (long) ((b[i + 1] & 0xff) << 16) | (long) ((b[i + 2] & 0xff) << 8) | (long) (b[i + 3] & 0xff) & 0xffffffffL;
            return n;
        }

        private void PUT_ULONG_BE(long n, byte[] b, int i) {
            b[i] = (byte) (int) (0xFF & n >> 24);
            b[i + 1] = (byte) (int) (0xFF & n >> 16);
            b[i + 2] = (byte) (int) (0xFF & n >> 8);
            b[i + 3] = (byte) (int) (0xFF & n);
        }

        private long SHL(long x, int n) {
            return (x & 0xFFFFFFFF) << n;
        }

        private long ROTL(long x, int n) {
            return SHL(x, n) | x >> (32 - n);
        }

        /**
         * ctx的sk交换。
         *
         * @param sk
         * @param i
         */
        private void SWAP(long[] sk, int i) {
            long t = sk[i];
            sk[i] = sk[(31 - i)];
            sk[(31 - i)] = t;
        }

        /**
         * S盒。
         * <p>
         * S盒是固定不变的。
         * <p>
         * The S-box is as follows:
         * 省略。
         * Note: substitution values for the byte xy (in hexadecimal format), e.g. when the input is 'EF', then the output is the value in row E and column F, i.e. 𝑆𝑏𝑜𝑥(𝐸𝐹) = 84.
         */
        public static final byte[] SBOX_TABLE = {(byte) 0xd6, (byte) 0x90, (byte) 0xe9, (byte) 0xfe,
                (byte) 0xcc, (byte) 0xe1, 0x3d, (byte) 0xb7, 0x16, (byte) 0xb6,
                0x14, (byte) 0xc2, 0x28, (byte) 0xfb, 0x2c, 0x05, 0x2b, 0x67,
                (byte) 0x9a, 0x76, 0x2a, (byte) 0xbe, 0x04, (byte) 0xc3,
                (byte) 0xaa, 0x44, 0x13, 0x26, 0x49, (byte) 0x86, 0x06,
                (byte) 0x99, (byte) 0x9c, 0x42, 0x50, (byte) 0xf4, (byte) 0x91,
                (byte) 0xef, (byte) 0x98, 0x7a, 0x33, 0x54, 0x0b, 0x43,
                (byte) 0xed, (byte) 0xcf, (byte) 0xac, 0x62, (byte) 0xe4,
                (byte) 0xb3, 0x1c, (byte) 0xa9, (byte) 0xc9, 0x08, (byte) 0xe8,
                (byte) 0x95, (byte) 0x80, (byte) 0xdf, (byte) 0x94, (byte) 0xfa,
                0x75, (byte) 0x8f, 0x3f, (byte) 0xa6, 0x47, 0x07, (byte) 0xa7,
                (byte) 0xfc, (byte) 0xf3, 0x73, 0x17, (byte) 0xba, (byte) 0x83,
                0x59, 0x3c, 0x19, (byte) 0xe6, (byte) 0x85, 0x4f, (byte) 0xa8,
                0x68, 0x6b, (byte) 0x81, (byte) 0xb2, 0x71, 0x64, (byte) 0xda,
                (byte) 0x8b, (byte) 0xf8, (byte) 0xeb, 0x0f, 0x4b, 0x70, 0x56,
                (byte) 0x9d, 0x35, 0x1e, 0x24, 0x0e, 0x5e, 0x63, 0x58, (byte) 0xd1,
                (byte) 0xa2, 0x25, 0x22, 0x7c, 0x3b, 0x01, 0x21, 0x78, (byte) 0x87,
                (byte) 0xd4, 0x00, 0x46, 0x57, (byte) 0x9f, (byte) 0xd3, 0x27,
                0x52, 0x4c, 0x36, 0x02, (byte) 0xe7, (byte) 0xa0, (byte) 0xc4,
                (byte) 0xc8, (byte) 0x9e, (byte) 0xea, (byte) 0xbf, (byte) 0x8a,
                (byte) 0xd2, 0x40, (byte) 0xc7, 0x38, (byte) 0xb5, (byte) 0xa3,
                (byte) 0xf7, (byte) 0xf2, (byte) 0xce, (byte) 0xf9, 0x61, 0x15,
                (byte) 0xa1, (byte) 0xe0, (byte) 0xae, 0x5d, (byte) 0xa4,
                (byte) 0x9b, 0x34, 0x1a, 0x55, (byte) 0xad, (byte) 0x93, 0x32,
                0x30, (byte) 0xf5, (byte) 0x8c, (byte) 0xb1, (byte) 0xe3, 0x1d,
                (byte) 0xf6, (byte) 0xe2, 0x2e, (byte) 0x82, 0x66, (byte) 0xca,
                0x60, (byte) 0xc0, 0x29, 0x23, (byte) 0xab, 0x0d, 0x53, 0x4e, 0x6f,
                (byte) 0xd5, (byte) 0xdb, 0x37, 0x45, (byte) 0xde, (byte) 0xfd,
                (byte) 0x8e, 0x2f, 0x03, (byte) 0xff, 0x6a, 0x72, 0x6d, 0x6c, 0x5b,
                0x51, (byte) 0x8d, 0x1b, (byte) 0xaf, (byte) 0x92, (byte) 0xbb,
                (byte) 0xdd, (byte) 0xbc, 0x7f, 0x11, (byte) 0xd9, 0x5c, 0x41,
                0x1f, 0x10, 0x5a, (byte) 0xd8, 0x0a, (byte) 0xc1, 0x31,
                (byte) 0x88, (byte) 0xa5, (byte) 0xcd, 0x7b, (byte) 0xbd, 0x2d,
                0x74, (byte) 0xd0, 0x12, (byte) 0xb8, (byte) 0xe5, (byte) 0xb4,
                (byte) 0xb0, (byte) 0x89, 0x69, (byte) 0x97, 0x4a, 0x0c,
                (byte) 0x96, 0x77, 0x7e, 0x65, (byte) 0xb9, (byte) 0xf1, 0x09,
                (byte) 0xc5, 0x6e, (byte) 0xc6, (byte) 0x84, 0x18, (byte) 0xf0,
                0x7d, (byte) 0xec, 0x3a, (byte) 0xdc, 0x4d, 0x20, 0x79,
                (byte) 0xee, 0x5f, 0x3e, (byte) 0xd7, (byte) 0xcb, 0x39, 0x48};

        /**
         * 系统参数FK的取值（取值固定）。
         * 4个。
         * <p>
         * (2) The system parameter 𝐹𝐾 is:
         * 𝐹𝐾0 = (𝐴3𝐵1𝐵𝐴𝐶6), 𝐹𝐾2 = (56𝐴𝐴3350),
         * 𝐹𝐾3 = (677𝐷9197), 𝐹𝐾4 = (𝐵27022𝐷𝐶).
         */
        public static final int[] FK = {0xa3b1bac6, 0x56aa3350, 0x677d9197, 0xb27022dc};

        /**
         * 固定参数CK。
         * 32个。
         * <p>
         * (3) The fixed parameter 𝐶𝐾 is used in the key expansion algorithm. Suppose 𝑐𝑘6,^ is
         * the j -th byte of 𝐶𝐾6(𝑖 = 0,1, ... ,31, 𝑗 = 0,1,2,3), i.e. 𝐶𝐾6 = (𝑐𝑘6,0, 𝑐𝑘6,2, 𝑐𝑘6,3, 𝑐𝑘6,4)) ∈ (𝑍M)C, then 𝑐𝑘 = (4𝑖 + 𝑗)×7(𝑚𝑜𝑑 256). To be specific, the values of the fixed
         * 3 6,^
         * parameters 𝐶𝐾6(𝑖=0,1,...,31) are:
         * 00070E15, 70777E85, E0E7EEF5, 50575E65, C0C7CED5, 30373E45, A0A7AEB5, 10171E25,
         * 1C232A31, 8C939AA1, FC030A11, 6C737A81, DCE3EAF1, 4C535A61, BCC3CAD1, 2C333A41,
         * 383F464D, A8AFB6BD, 181F262D, 888F969D, F8FF060D, 686F767D, D8DFE6ED, 484F565D,
         * 545B6269, C4CBD2D9, 343B4249, A4ABB2B9, 141B2229, 848B9299, F4FB0209, 646B7279.
         */
        public static final int[] CK = {0x00070e15, 0x1c232a31, 0x383f464d, 0x545b6269,
                0x70777e85, 0x8c939aa1, 0xa8afb6bd, 0xc4cbd2d9,
                0xe0e7eef5, 0xfc030a11, 0x181f262d, 0x343b4249,
                0x50575e65, 0x6c737a81, 0x888f969d, 0xa4abb2b9,
                0xc0c7ced5, 0xdce3eaf1, 0xf8ff060d, 0x141b2229,
                0x30373e45, 0x4c535a61, 0x686f767d, 0x848b9299,
                0xa0a7aeb5, 0xbcc3cad1, 0xd8dfe6ed, 0xf4fb0209,
                0x10171e25, 0x2c333a41, 0x484f565d, 0x646b7279};

        private byte sm4Sbox(byte inch) {
            int i = inch & 0xFF;
            byte retVal = SBOX_TABLE[i];
            return retVal;
        }

        /**
         * 组合T。
         * 包括：
         * （1）非线性变换𝝉(tau)
         * （2）线性变换L。
         * <p>
         * 6.2 Permutation 𝑻
         * 𝑇: 𝑍43 → 𝑍43 is an invertible transformation, composed of a nonlinear transformation 𝜏 33
         * and a linear transformation 𝐿. That is, 𝑇 ∙ = 𝐿(𝜏(∙)).
         * <p>
         * (1) Nonlinear transformation 𝝉
         * 𝜏 is composed of 4 S-boxes in parallel. Suppose 𝐴 = (𝑎 ,𝑎 ,𝑎 ,𝑎 ) ∈ (𝑍M)C is input to 0234 3
         * 𝜏, and 𝐵 = (𝑏 ,𝑏 ,𝑏 ,𝑏 ) ∈ (𝑍M)C is the corresponding output, then 0234 3
         * 𝑏0,𝑏2,𝑏3,𝑏4 = 𝜏 𝐴 = (𝑆𝑏𝑜𝑥 𝑎0 ,𝑆𝑏𝑜𝑥 𝑎2 ,𝑆𝑏𝑜𝑥 𝑎3 ,𝑆𝑏𝑜𝑥 𝑎4 ).
         * The S-box is as follows:
         *
         * @param ka
         * @return
         * @see SM4Core#SBOX_TABLE
         * <p>
         * (2) Linear transformation 𝑳
         * The output from the nonlinear transformation 𝜏 is the input to the linear
         * transformation 𝐿. Suppose the input to 𝐿 is 𝐵 ∈ 𝑍43, and the corresponding output is 3
         * 𝐶 ∈ 𝑍43, then 3
         * 𝐶=𝐿 𝐵 =𝐵⊕(𝐵<<<2)⊕(𝐵<<<10)⊕(𝐵<<<18)⊕(𝐵<<<24).
         */
        private long sm4Lt(long ka) {
            long bb = 0L;
            long c = 0L;
            byte[] a = new byte[4];
            byte[] b = new byte[4];
            PUT_ULONG_BE(ka, a, 0);
            b[0] = sm4Sbox(a[0]);
            b[1] = sm4Sbox(a[1]);
            b[2] = sm4Sbox(a[2]);
            b[3] = sm4Sbox(a[3]);
            bb = GET_ULONG_BE(b, 0);
            c = bb ^ ROTL(bb, 2) ^ ROTL(bb, 10) ^ ROTL(bb, 18) ^ ROTL(bb, 24);
            return c;
        }

        // X（r+1）、X（r+2）、X（r+3）与轮密钥进行异或运算

        /**
         * 轮函数F。
         * <p>
         * 6 Round Function 𝑭
         * 6.1 Round Function Structure
         * Suppose the input to round function is (𝑋0 ,𝑋1 ,𝑋2 ,𝑋3 ) ∈ (𝑍43)C and the round key is
         * 𝑟𝑘 ∈ 𝑍43, then 𝐹 can be represented as: 3
         * 𝐹 𝑋0,𝑋2,𝑋3,𝑋4,𝑟𝑘   = 𝑋0 ⊕ 𝑇(𝑋2 ⊕ 𝑋3 ⊕ 𝑋4 ⊕ 𝑟𝑘).
         *
         * @param x0
         * @param x1
         * @param x2
         * @param x3
         * @param rk
         * @return
         */
        private long sm4F(long x0, long x1, long x2, long x3, long rk) {
            return x0 ^ sm4Lt(x1 ^ x2 ^ x3 ^ rk);
        }

        private long sm4CalciRK(long ka) {
            long bb = 0L;
            long rk = 0L;
            byte[] a = new byte[4];
            byte[] b = new byte[4];
            PUT_ULONG_BE(ka, a, 0);
            b[0] = sm4Sbox(a[0]);
            b[1] = sm4Sbox(a[1]);
            b[2] = sm4Sbox(a[2]);
            b[3] = sm4Sbox(a[3]);
            bb = GET_ULONG_BE(b, 0);
            rk = bb ^ ROTL(bb, 13) ^ ROTL(bb, 23);
            return rk;
        }

        /**
         * 设置SK。
         *
         * @param SK
         * @param key
         */
        private void setKey(long[] SK, byte[] key) {
            long[] MK = new long[4];
            long[] k = new long[36];
            int i = 0;
            MK[0] = GET_ULONG_BE(key, 0);
            MK[1] = GET_ULONG_BE(key, 4);
            MK[2] = GET_ULONG_BE(key, 8);
            MK[3] = GET_ULONG_BE(key, 12);
            k[0] = MK[0] ^ (long) FK[0];
            k[1] = MK[1] ^ (long) FK[1];
            k[2] = MK[2] ^ (long) FK[2];
            k[3] = MK[3] ^ (long) FK[3];
            for (; i < 32; i++) {
                k[(i + 4)] = (k[i] ^ sm4CalciRK(k[(i + 1)] ^ k[(i + 2)] ^ k[(i + 3)] ^ (long) CK[i]));
                SK[i] = k[(i + 4)];
            }
        }


        private void sm4_one_round(long[] sk, byte[] input, byte[] output) {
            int i = 0;
            long[] ulbuf = new long[36];
            ulbuf[0] = GET_ULONG_BE(input, 0);
            ulbuf[1] = GET_ULONG_BE(input, 4);
            ulbuf[2] = GET_ULONG_BE(input, 8);
            ulbuf[3] = GET_ULONG_BE(input, 12);

            // 进行32轮的加密变换操作
            while (i < 32) {
                ulbuf[(i + 4)] = sm4F(ulbuf[i], ulbuf[(i + 1)], ulbuf[(i + 2)], ulbuf[(i + 3)], sk[i]);
                i++;
            }

            PUT_ULONG_BE(ulbuf[35], output, 0);
            PUT_ULONG_BE(ulbuf[34], output, 4);
            PUT_ULONG_BE(ulbuf[33], output, 8);
            PUT_ULONG_BE(ulbuf[32], output, 12);
        }

        /**
         * 填充。
         *
         * @param input 需要填充的明文输入字节，或者复原被填充的解密后明文输出字节
         * @param mode  加密还是解密
         * @return 填充后的字节
         */
        private byte[] padding(byte[] input, int mode) {
            if (input == null) {
                return null;
            }

            byte[] ret = (byte[]) null;
            if (mode == SM4_ENCRYPT) {
                int p = 16 - input.length % 16;
                ret = new byte[input.length + p];
                System.arraycopy(input, 0, ret, 0, input.length);
                for (int i = 0; i < p; i++) {
                    ret[input.length + i] = (byte) p;
                }
            } else {
                int p = input[input.length - 1];
                ret = new byte[input.length - p];
                System.arraycopy(input, 0, ret, 0, input.length - p);
            }
            return ret;
        }

        /**
         * 加密设置key。
         * 设置ctx是加密还是解密模式，设置ctx的sk（key）。
         *
         * @param ctx SM4上下文
         * @param key 密钥
         * @throws Exception ctx为空，或者密钥不是16位。
         */
        public void sm4_setkey_enc(SM4Context ctx, byte[] key) throws Exception {
            if (ctx == null) {
                throw new Exception("ctx is null!");
            }

            if (key == null || key.length != 16) {
                throw new Exception("key error!");
            }

            ctx.mode = SM4_ENCRYPT;
            setKey(ctx.sk, key);
        }

        /**
         * 解密设置key。
         *
         * @param ctx
         * @param key
         * @throws Exception
         */
        public void sm4_setkey_dec(SM4Context ctx, byte[] key) throws Exception {
            if (ctx == null) {
                throw new Exception("ctx is null!");
            }

            if (key == null || key.length != 16) {
                throw new Exception("key error!");
            }

            int i = 0;
            ctx.mode = SM4_DECRYPT;
            setKey(ctx.sk, key);
            for (i = 0; i < 16; i++) {
                SWAP(ctx.sk, i);
            }
        }

        public byte[] sm4_crypt_ecb(SM4Context ctx, byte[] input) throws Exception {
            if (input == null) {
                throw new Exception("input is null!");
            }

            // 填充
            if ((ctx.isPadding) && (ctx.mode == SM4_ENCRYPT)) {
                input = padding(input, SM4_ENCRYPT);
            }

            int length = input.length;
            ByteArrayInputStream bins = new ByteArrayInputStream(input);
            ByteArrayOutputStream bous = new ByteArrayOutputStream();

            for (; length > 0; length -= 16) {
                byte[] in = new byte[16];
                byte[] out = new byte[16];
                bins.read(in);
                sm4_one_round(ctx.sk, in, out);
                bous.write(out);
            }

            byte[] output = bous.toByteArray();
            if (ctx.isPadding && ctx.mode == SM4_DECRYPT) {
                output = padding(output, SM4_DECRYPT);
            }
            bins.close();
            bous.close();
            return output;
        }

        /**
         * @param ctx
         * @param iv    初始化向量，必须为16位
         * @param input
         * @return
         * @throws Exception
         */
        public byte[] sm4_crypt_cbc(SM4Context ctx, byte[] iv, byte[] input) throws Exception {
            if (iv == null || iv.length != 16) {
                throw new Exception("iv error!初始化向量不能为空且必须为16位");
            }

            if (input == null) {
                throw new Exception("input is null!");
            }

            // 如果加密，且需要填充
            if (ctx.isPadding && ctx.mode == SM4_ENCRYPT) {
                input = padding(input, SM4_ENCRYPT);
            }

            int i = 0;
            int length = input.length;
            ByteArrayInputStream bins = new ByteArrayInputStream(input);
            ByteArrayOutputStream bous = new ByteArrayOutputStream();
            if (ctx.mode == SM4_ENCRYPT) {
                for (; length > 0; length -= 16) {
                    byte[] in = new byte[16];
                    byte[] out = new byte[16];
                    byte[] out1 = new byte[16];

                    bins.read(in);
                    for (i = 0; i < 16; i++) {
                        out[i] = ((byte) (in[i] ^ iv[i]));
                    }
                    sm4_one_round(ctx.sk, out, out1);
                    System.arraycopy(out1, 0, iv, 0, 16);
                    bous.write(out1);
                }
            } else {
                byte[] temp = new byte[16];
                for (; length > 0; length -= 16) {
                    byte[] in = new byte[16];
                    byte[] out = new byte[16];
                    byte[] out1 = new byte[16];

                    bins.read(in);
                    System.arraycopy(in, 0, temp, 0, 16);
                    sm4_one_round(ctx.sk, in, out);
                    for (i = 0; i < 16; i++) {
                        out1[i] = ((byte) (out[i] ^ iv[i]));
                    }
                    System.arraycopy(temp, 0, iv, 0, 16);
                    bous.write(out1);
                }
            }

            byte[] output = bous.toByteArray();
            if (ctx.isPadding && ctx.mode == SM4_DECRYPT) {
                output = padding(output, SM4_DECRYPT);
            }
            bins.close();
            bous.close();
            return output;
        }
    }

    /**
     * SM4上下文。
     */
    private static class SM4Context {
        /**
         * 加密还是解密
         */
        public int mode;
        /**
         * 密钥
         */
        public long[] sk;

        /**
         * 是否填充
         */
        public boolean isPadding;

        public SM4Context() {
            this.mode = 1;
            this.isPadding = true;
            this.sk = new long[32];
        }
    }

}
