/**
 * Copyright(c) Foresee Science & Technology Ltd.
 */
package com.foresee.open.sdk.kit.gm;

import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.DataLengthException;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.*;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.SecureRandom;

/**
 * <pre>
 * SM2工具类，支持加密、解密、签名、验签
 * </pre>
 *
 * @author chenqiang@foresee.com.cn
 * @version 1.00.00
 * @date 2020年6月8日
 */
public class SM2Util {

    /**
     * 加密结果拼接模式，默认是C1C3C2
     */
    public static Mode mode = Mode.C1C3C2;

    public enum Mode {
        C1C2C3, C1C3C2;
    }

    /**
     * 公钥私钥密钥对
     */
    public static class KeyPair {

        private String privateKey;

        private String publicKey;


        public KeyPair(String privateKey, String publicKey) {
            super();
            this.privateKey = privateKey;
            this.publicKey = publicKey;
        }

        public String getPrivateKey() {
            return privateKey;
        }

        public void setPrivateKey(String privateKey) {
            this.privateKey = privateKey;
        }

        public String getPublicKey() {
            return publicKey;
        }

        public void setPublicKey(String publicKey) {
            this.publicKey = publicKey;
        }

    }


    /**
     * 生成SM2私钥公钥对
     *
     * @return KeyPair
     */
    public static KeyPair generateKeyPair() {
        SM2 sm2 = SM2.Instance();
        AsymmetricCipherKeyPair key = sm2.ecc_key_pair_generator.generateKeyPair();
        ECPrivateKeyParameters ecpriv = (ECPrivateKeyParameters) key.getPrivate();
        ECPublicKeyParameters ecpub = (ECPublicKeyParameters) key.getPublic();
        BigInteger privateKey = ecpriv.getD();
        ECPoint publicKey = ecpub.getQ();

        return new KeyPair(ByteUtil.byteToHex(privateKey.toByteArray()), ByteUtil.byteToHex(publicKey.getEncoded(false)));
    }

    /**
     * 根据SM2私钥生成公钥
     *
     * @param privateKey 十六进制私钥
     * @return 十六进制公钥
     */
    public static String generatePublicKey(String privateKey) {
        BigInteger d = new BigInteger(ByteUtil.hexToByte(privateKey));
        ECPoint ecPoint = SM2.ecc_bc_spec.getG().multiply(d);
        ECPublicKeyParameters publicParams = new ECPublicKeyParameters(ecPoint, SM2.ecc_bc_spec);
        return ByteUtil.byteToHex(publicParams.getQ().getEncoded(false));
    }

    /**
     * 使用公钥加密
     *
     * @param publicKey 十六进制公钥
     * @param data      明文数据
     * @return 十六进制密文
     */
    public static String encrypt(String publicKey, String data) {
        return encrypt(ByteUtil.hexToByte(publicKey), data.getBytes());
    }

    /**
     * 使用公钥加密
     *
     * @param publicKey 公钥
     * @param data      明文数据
     * @return 十六进制密文
     */
    public static String encrypt(byte[] publicKey, byte[] data) {
        if (publicKey == null || publicKey.length == 0) {
            return null;
        }

        if (data == null || data.length == 0) {
            return null;
        }

        byte[] source = new byte[data.length];
        System.arraycopy(data, 0, source, 0, data.length);

        Cipher cipher = new Cipher();
        SM2 sm2 = SM2.Instance();
        ECPoint userKey = SM2.ecc_curve.decodePoint(publicKey);

        ECPoint c1 = cipher.initEnc(sm2, userKey);
        cipher.encrypt(source);
        byte[] c3 = new byte[32];
        cipher.doFinal(c3);

        if (mode == Mode.C1C3C2) {
            return ByteUtil.byteToHex(ByteUtil.combineBytes(c1.getEncoded(false), c3, source));
        } else {
            return ByteUtil.byteToHex(ByteUtil.combineBytes(c1.getEncoded(false), source, c3));
        }

    }

    /**
     * 使用私钥解密
     *
     * @param privateKey 十六进制私钥
     * @param cipherText 十六进制密文
     * @return 明文数据
     */
    public static String decrypt(String privateKey, String cipherText) {
        return new String(decrypt(ByteUtil.hexToByte(privateKey), ByteUtil.hexToByte(cipherText)));
    }

    /**
     * 使用私钥解密
     *
     * @param privateKey 私钥
     * @param cipherText 密文
     * @return 明文数据字节数组
     */
    public static byte[] decrypt(byte[] privateKey, byte[] cipherText) {
        if (privateKey == null || privateKey.length == 0) {
            return null;
        }

        if (cipherText == null || cipherText.length == 0) {
            return null;
        }

        //加密字节数组转换为十六进制的字符，长度变为cipherText.length * 2  
        String data = ByteUtil.byteToHex(cipherText);

        // 分解加密字串 （C1 = C1标志位2位 + C1实体部分128位 = 130位 （C3 = C3实体部分64位 = 64位 （C2 = encryptedData.length * 2 - C1长度 - C3长度位
        byte[] c1Bytes = ByteUtil.hexToByte(data.substring(0, 130));
        int c2Len = cipherText.length - 97;

        byte[] c2;
        byte[] c3;

        if (mode == Mode.C1C3C2) {
            c3 = ByteUtil.hexToByte(data.substring(130, 194));
            c2 = ByteUtil.hexToByte(data.substring(194, 194 + 2 * c2Len));

        } else {
            c2 = ByteUtil.hexToByte(data.substring(130, 130 + 2 * c2Len));
            c3 = ByteUtil.hexToByte(data.substring(130 + 2 * c2Len, 194 + 2 * c2Len));
        }

        BigInteger userD = new BigInteger(1, privateKey);

        //通过C1实体字节来生成ECPoint  
        ECPoint c1 = SM2.ecc_curve.decodePoint(c1Bytes);
        Cipher cipher = new Cipher();
        cipher.initDec(userD, c1);
        cipher.decrypt(c2);
        cipher.doFinal(c3);

        //返回解密结果  
        return c2;
    }

    /**
     * SM2 私钥签名
     *
     * @param privateKey 十六进制私钥
     * @param data       源数据
     * @return 签名值
     */
    public static String sign(String privateKey, String data) {
        return sign(privateKey, data, null);
    }

    /**
     * SM2 私钥签名
     *
     * @param privateKey 十六进制私钥
     * @param data       源数据字节数据
     * @return 签名值字节数组
     */
    public static byte[] sign(String privateKey, byte[] data) {
        return sign(privateKey, data, null);
    }

    /**
     * SM2 私钥签名
     *
     * @param privateKey 十六进制私钥
     * @param data       源数据
     * @param userId     用户标识
     * @return 签名值
     */
    public static String sign(String privateKey, String data, String userId) {
        byte[] bytes = sign(privateKey, data.getBytes(), userId == null ? null : userId.getBytes());
        if (bytes != null) {
            return ByteUtil.byteToHex(bytes);
        }

        return null;
    }

    /**
     * SM2 私钥签名
     *
     * @param privateKey 十六进制私钥
     * @param data       源数据字节数据
     * @param userId     用户标识字节数组
     * @return 签名值字节数据
     */
    public static byte[] sign(String privateKey, byte[] data, byte[] userId) {
        CipherParameters prikey = createPrivateCipherParameters(privateKey);

        return sign(prikey, data, userId);
    }

    /**
     * SM2 私钥签名
     *
     * @param param  私钥参数
     * @param data   源数据字节数据
     * @param userId 用户标识字节数组
     * @return 签名值字节数据
     */
    public static byte[] sign(CipherParameters param, byte[] data, byte[] userId) {
        CipherParameters priKey = param;
        // 先做一次Hash
        if (priKey instanceof ParametersWithRandom) {
            ParametersWithRandom rParam = (ParametersWithRandom) param;
            priKey = rParam.getParameters();
        }

        if (!(priKey instanceof ECPrivateKeyParameters)) {
            throw new IllegalArgumentException("EC private key required for signing");
        }

        SM3Digest digest = new SM3Digest();
        byte[] e = new byte[digest.getDigestSize()];

        if (userId != null && userId.length > 0) {
            // e=Hv(M_), M_=ZA||M
            byte[] za = null;
            za = calculateZA((ECPrivateKeyParameters) priKey, userId);
            digest.update(za, 0, za.length);
        }

        digest.update(data, 0, data.length);
        digest.doFinal(e, 0);

        // 签名,e 为 Hv(ZA||M)
        BigInteger[] sig;
        Sm2Signer signer = new Sm2Signer();

        try {
            signer.init(true, param);
            sig = signer.generateSignature(e);
        } catch (Exception ex) {
            throw new RuntimeException("Sign fail: " + ex.getMessage(), ex);
        }

        // 拼接RS
        byte[] signature;
        signature = ByteUtil.combineBytes(ByteUtil.bigIntegerToByteArray(sig[0]), ByteUtil.bigIntegerToByteArray(sig[1]));

        return signature;
    }

    /**
     * SM2公钥验签
     *
     * @param publicKey 十六进制公钥
     * @param data      源数据字节数组
     * @param sign      签名值字节数组
     * @return 是否匹配
     */
    public static boolean verify(String publicKey, byte[] data, byte[] sign) {
        CipherParameters pubKey = createPublicCipherParameters(publicKey);
        return verify(pubKey, data, null, sign);
    }

    /**
     * SM2公钥验签
     *
     * @param publicKey 十六进制公钥
     * @param data      源数据
     * @param sign      签名值
     * @return 是否匹配
     */
    public static boolean verify(String publicKey, String data, String sign) {
        return verify(publicKey, data, null, sign);
    }

    /**
     * SM2公钥验签
     *
     * @param publicKey 十六进制公钥
     * @param data      源数据
     * @param userId    用户标识
     * @param sign      签名值
     * @return 是否匹配
     */
    public static boolean verify(String publicKey, String data, String userId, String sign) {
        return verify(publicKey, data.getBytes(), userId == null ? null : userId.getBytes(), ByteUtil.hexToByte(sign));
    }

    /**
     * SM2公钥验签
     *
     * @param publicKey 十六进制公钥
     * @param data      源数据字节数组
     * @param userId    用户标识字节数组
     * @param sign      签名值字节数组
     * @return 是否匹配
     */
    public static boolean verify(String publicKey, byte[] data, byte[] userId, byte[] sign) {
        CipherParameters pubKey = createPublicCipherParameters(publicKey);
        return verify(pubKey, data, userId, sign);
    }

    /**
     * SM2公钥验签
     *
     * @param data   源数据字节数组
     * @param userId 用户标识字节数组
     * @param sign   签名值字节数组
     * @return 是否匹配
     */
    public static boolean verify(CipherParameters pubKey, byte[] data, byte[] userId, byte[] sign) {

        SM3Digest digest = new SM3Digest();
        byte[] e = new byte[digest.getDigestSize()];

        if (userId != null && userId.length > 0) {
            // e=Hv(M_), M_=ZA||M
            byte[] za = null;
            za = calculateZA((ECPublicKeyParameters) pubKey, userId);
            digest.update(za, 0, za.length);
        }

        digest.update(data, 0, data.length);
        digest.doFinal(e, 0);

        BigInteger r;
        BigInteger s;

        InputStream is = new ByteArrayInputStream(sign);
        byte[] rb = new byte[sign.length / 2];
        byte[] sb = new byte[sign.length / 2];
        try {
            is.read(rb, 0, sign.length / 2);
            is.read(sb, 0, sign.length / 2);
        } catch (IOException e1) {
            return false;
        }

        r = new BigInteger(1, rb);
        s = new BigInteger(1, sb);

        Sm2Signer signer = new Sm2Signer();
        try {
            signer.init(false, pubKey);
        } catch (InvalidKeyException ex) {
            throw new RuntimeException("Invalid public key forverify.");
        }

        return signer.verifySignature(e, r, s);
    }

    /// 生成SM2私钥
    public static ECPrivateKeyParameters createPrivateCipherParameters(String privateKey) {
        BigInteger d = new BigInteger(1, ByteUtil.hexToByte(privateKey));
        return new ECPrivateKeyParameters(d, SM2.ecc_bc_spec);
    }

    public static ECPublicKeyParameters createPublicCipherParameters(String publicKey) {
        byte[] publicKeyBytes = ByteUtil.hexToByte(publicKey);
        byte[] encodedPublicKey = publicKeyBytes;
        byte[] ecP = new byte[64];
        System.arraycopy(encodedPublicKey, 1, ecP, 0, ecP.length);
        byte[] certPKX = new byte[32];
        byte[] certPKY = new byte[32];
        System.arraycopy(ecP, 0, certPKX, 0, 32);
        System.arraycopy(ecP, 32, certPKY, 0, 32);
        ECPublicKeyParameters pubKey = createPublicKey(certPKX, certPKY);
        return pubKey;
    }

    public static ECPublicKeyParameters createPublicKey(byte[] xH, byte[] yH) {
        BigInteger qx = new BigInteger(1, xH);
        BigInteger qy = new BigInteger(1, yH);
        ECPoint q = createPoint(qx, qy);
        return new ECPublicKeyParameters(q, SM2.ecc_bc_spec);
    }

    public static ECPoint createPoint(BigInteger x, BigInteger y) {
        ECCurve curve = new ECCurve.Fp(SM2.ecc_p, SM2.ecc_a, SM2.ecc_b);
        return curve.createPoint(x, y);
    }

    /**
     * ZA: 用户的标志、部分椭圆曲线参数、用户公钥的 杂凑值
     *
     * @param G
     * @param a
     * @param b
     * @param A
     * @param id
     * @return
     */
    private static byte[] calculateZA(ECPoint G, BigInteger a, BigInteger b, ECPoint A, byte[] id) {
        SM3Digest digest = new SM3Digest();
        byte[] e = new byte[digest.getDigestSize()];
        byte[] tmp = null;

        // ENTLA
        int entla = id.length * 8;
        digest.update((byte) ((entla >> 8) & 0xFF));
        digest.update((byte) (entla & 0xFF));

        // IDA
        digest.update(id, 0, id.length);

        // a
        tmp = ByteUtil.bigIntegerToByteArray(a);
        digest.update(tmp, 0, tmp.length);

        // b
        tmp = ByteUtil.bigIntegerToByteArray(b);
        digest.update(tmp, 0, tmp.length);

        // xG
        tmp = ByteUtil.bigIntegerToByteArray(G.getAffineXCoord().toBigInteger());
        digest.update(tmp, 0, tmp.length);

        // yG
        tmp = ByteUtil.bigIntegerToByteArray(G.getAffineYCoord().toBigInteger());
        digest.update(tmp, 0, tmp.length);

        // xA
        tmp = ByteUtil.bigIntegerToByteArray(A.getAffineXCoord().toBigInteger());
        digest.update(tmp, 0, tmp.length);

        // yA
        tmp = ByteUtil.bigIntegerToByteArray(A.getAffineYCoord().toBigInteger());
        digest.update(tmp, 0, tmp.length);

        digest.doFinal(e, 0);
        return e;
    }

    public static byte[] calculateZA(ECPrivateKeyParameters priKey, byte[] id) {
        ECPoint G = priKey.getParameters().getG();
        BigInteger a = priKey.getParameters().getCurve().getA().toBigInteger();
        BigInteger b = priKey.getParameters().getCurve().getB().toBigInteger();

        ECPoint A = ECAlgorithms.referenceMultiply(G, priKey.getD()).normalize();

        return calculateZA(G, a, b, A, id);
    }

    public static byte[] calculateZA(ECPublicKeyParameters pubKey, byte[] id) {

        ECPoint G = pubKey.getParameters().getG();
        BigInteger a = pubKey.getParameters().getCurve().getA().toBigInteger();
        BigInteger b = pubKey.getParameters().getCurve().getB().toBigInteger();

        ECPoint A = pubKey.getQ();

        return calculateZA(G, a, b, A, id);
    }


}

class SM2 {

    public static String[] ecc_param = {"FFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFFFFFFFFFF",
            "FFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFFFFFFFFFC",
            "28E9FA9E9D9F5E344D5A9E4BCF6509A7F39789F515AB8F92DDBCBD414D940E93",
            "FFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFF7203DF6B21C6052B53BBF40939D54123",
            "32C4AE2C1F1981195F9904466A39C9948FE30BBFF2660BE1715A4589334C74C7",
            "BC3736A2F4F6779C59BDCEE36B692153D0A9877CC62A474002DF32E52139F0A0"};


    public static SM2 Instance() {
        return new SM2();
    }


    public static BigInteger ecc_p;

    public static BigInteger ecc_a;

    public static BigInteger ecc_b;

    public static BigInteger ecc_n;

    public static BigInteger ecc_gx;

    public static BigInteger ecc_gy;

    public static ECCurve ecc_curve;

    public static ECPoint ecc_point_g;

    public static ECDomainParameters ecc_bc_spec;

    public final ECKeyPairGenerator ecc_key_pair_generator;

    static {
        ecc_p = new BigInteger(ecc_param[0], 16);
        ecc_a = new BigInteger(ecc_param[1], 16);
        ecc_b = new BigInteger(ecc_param[2], 16);
        ecc_n = new BigInteger(ecc_param[3], 16);
        ecc_gx = new BigInteger(ecc_param[4], 16);
        ecc_gy = new BigInteger(ecc_param[5], 16);

        ecc_curve = new ECCurve.Fp(ecc_p, ecc_a, ecc_b);
        ecc_point_g = ecc_curve.createPoint(ecc_gx, ecc_gy);
        ecc_bc_spec = new ECDomainParameters(ecc_curve, ecc_point_g, ecc_n);
    }


    public SM2() {
        ECKeyGenerationParameters ecc_ecgenparam;
        ecc_ecgenparam = new ECKeyGenerationParameters(this.ecc_bc_spec, new SecureRandom());

        this.ecc_key_pair_generator = new ECKeyPairGenerator();
        this.ecc_key_pair_generator.init(ecc_ecgenparam);
    }
}

class Cipher {

    private int ct;

    private ECPoint p2;

    private SM3Digest sm3keybase;

    private SM3Digest sm3c3;

    private byte key[];

    private byte keyOff;


    public Cipher() {
        this.ct = 1;
        this.key = new byte[32];
        this.keyOff = 0;
    }

    private void reset() {
        this.sm3keybase = new SM3Digest();
        this.sm3c3 = new SM3Digest();

        byte p[] = ByteUtil.byteConvert32Bytes(p2.normalize().getXCoord().toBigInteger());
        this.sm3keybase.update(p, 0, p.length);
        this.sm3c3.update(p, 0, p.length);

        p = ByteUtil.byteConvert32Bytes(p2.normalize().getYCoord().toBigInteger());
        this.sm3keybase.update(p, 0, p.length);
        this.ct = 1;
        nextKey();
    }

    private void nextKey() {
        SM3Digest sm3keycur = new SM3Digest(this.sm3keybase);
        sm3keycur.update((byte) (ct >> 24 & 0xff));
        sm3keycur.update((byte) (ct >> 16 & 0xff));
        sm3keycur.update((byte) (ct >> 8 & 0xff));
        sm3keycur.update((byte) (ct & 0xff));
        sm3keycur.doFinal(key, 0);
        this.keyOff = 0;
        this.ct++;
    }

    public ECPoint initEnc(SM2 sm2, ECPoint userKey) {
        AsymmetricCipherKeyPair key = sm2.ecc_key_pair_generator.generateKeyPair();
        ECPrivateKeyParameters ecpriv = (ECPrivateKeyParameters) key.getPrivate();
        ECPublicKeyParameters ecpub = (ECPublicKeyParameters) key.getPublic();
        BigInteger k = ecpriv.getD();
        ECPoint c1 = ecpub.getQ();
        this.p2 = userKey.multiply(k);
        reset();
        return c1;
    }

    public void encrypt(byte data[]) {
        this.sm3c3.update(data, 0, data.length);
        for (int i = 0; i < data.length; i++) {
            if (keyOff == key.length) {
                nextKey();
            }
            data[i] ^= key[keyOff++];
        }
    }

    public void initDec(BigInteger userD, ECPoint c1) {
        this.p2 = c1.multiply(userD);
        reset();
    }

    public void decrypt(byte data[]) {
        for (int i = 0; i < data.length; i++) {
            if (keyOff == key.length) {
                nextKey();
            }
            data[i] ^= key[keyOff++];
        }

        this.sm3c3.update(data, 0, data.length);
    }

    public void doFinal(byte c3[]) {
        byte p[] = ByteUtil.byteConvert32Bytes(p2.normalize().getYCoord().toBigInteger());
        this.sm3c3.update(p, 0, p.length);
        this.sm3c3.doFinal(c3, 0);
        reset();
    }
}

class Sm2Signer {

    private Boolean isSign;

    private ECKeyParameters key;

    private SecureRandom random;


    public String getAlgorithmName() {
        return "Sm2Signer";
    }

    public void init(boolean isSign, CipherParameters parameters) throws InvalidKeyException {
        this.isSign = isSign;
        if (isSign)// 签名
        {
            if (parameters instanceof ParametersWithRandom) {
                ParametersWithRandom rParam = (ParametersWithRandom) parameters;
                this.random = rParam.getRandom();
                parameters = rParam.getParameters();
            } else {
                this.random = new SecureRandom();
            }

            if (!(parameters instanceof ECPrivateKeyParameters)) {
                throw new InvalidKeyException("EC private key required for signing");
            }
            this.key = (ECPrivateKeyParameters) parameters;
        } else {// 验签
            if (!(parameters instanceof ECPublicKeyParameters)) {
                throw new InvalidKeyException("EC public key required for verification");
            }
            this.key = (ECPublicKeyParameters) parameters;
        }
    }

    public BigInteger[] generateSignature(byte[] message) throws Exception {
        if (!this.isSign) {
            throw new Exception("not initialised for signing");
        }
        // 阶n
        BigInteger n = ((ECPrivateKeyParameters) this.key).getParameters().getN();
        int nBitLength = n.bitLength();
        // M_=ZA||M，e=Hv(M_)
        BigInteger e = new BigInteger(1, message);
        int eBitLength = e.bitLength();

        ECPrivateKeyParameters privKey = (ECPrivateKeyParameters) key;
        if (eBitLength > nBitLength) {
            throw new DataLengthException("input too large for ECNR key.");
        }
        BigInteger r = null;
        BigInteger s = null;
        do {//判断s
            BigInteger k;// 随机数 k∈[1,n-1], (x1,y1)=[k]G
            AsymmetricCipherKeyPair tempPair;
            do {// r=(e+x1) mod n
                //generate another, but very temporary, key pair using, the same EC parameters
                //生成随机数k，及(x1,y1),借用用户密钥对方式生成：PA=[dA]G={xA,yA}
                ECKeyPairGenerator keyGen = new ECKeyPairGenerator();
                keyGen.init(new ECKeyGenerationParameters(privKey.getParameters(), this.random));
                tempPair = keyGen.generateKeyPair();
                k = ((ECPrivateKeyParameters) tempPair.getPrivate()).getD();
                ECPublicKeyParameters V = (ECPublicKeyParameters) tempPair.getPublic();
                BigInteger Vx = V.getQ().getAffineXCoord().toBigInteger();
                r = Vx.add(e).mod(n);
            } while (r.signum() == 0 || r.equals(BigInteger.ZERO) || r.add(k).equals(n));
            // s=((1/(1+dA)·(k-r·dA)) mod n
            BigInteger dA = privKey.getD(); // private key value

            BigInteger tmp = dA.add(BigInteger.ONE).modInverse(n);// 1/(1+dA)
            BigInteger tmp2 = k.subtract(r.multiply(dA));// (k-r·dA)
            s = tmp.multiply(tmp2).mod(n);
        } while (s.equals(BigInteger.ZERO));
        return new BigInteger[]{r, s};
    }

    public boolean verifySignature(byte[] message, BigInteger r, BigInteger s) {

        if (this.isSign) {
            throw new RuntimeException("Not initialised for verifying");
        }

        ECPublicKeyParameters pubKey = (ECPublicKeyParameters) key;
        // 阶n
        BigInteger n = pubKey.getParameters().getN();
        int nBitLength = n.bitLength();
        // M_'=ZA||M'，e'=Hv(M_')
        BigInteger e = new BigInteger(1, message);

        int eBitLength = e.bitLength();
        if (eBitLength > nBitLength) {
            throw new DataLengthException("Input too large for ECNR key.");
        }
        // r'∈[1,n-1]
        if (r.compareTo(BigInteger.ONE) < 0 || r.compareTo(n) >= 0) {
            return false;
        }
        // s' ∈ [1,n-1] NB: ECNR spec says 0
        if (s.compareTo(BigInteger.ONE) < 0 || s.compareTo(n) >= 0) {
            return false;
        }

        // t=(r'+s') mod n
        BigInteger t = r.add(s).mod(n);
        //不用判断t?
        if (t.compareTo(BigInteger.ZERO) == 0) {
            return false;
        }
        // (x1',y1')=[s']G+[t]PA, compute P = sG + rW
        ECPoint G = pubKey.getParameters().getG();
        ECPoint W = pubKey.getQ();
        // calculate P using Bouncy math
        ECPoint P = ECAlgorithms.sumOfTwoMultiplies(G, s, W, t).normalize();
        if (P.isInfinity()) {
            return false;
        }
        //R=(e'+x1') mod n
        BigInteger x = P.getAffineXCoord().toBigInteger();
        BigInteger R = e.add(x).mod(n);
        return R.equals(r);
    }
}