/*
 * <p>Copyright Foresee Science & Technology Co.</p>
 * 
 * @author <a href="mailto:chenqiang@foresee.com.cn">chenqiang</a> $Id: ZIPUtil.java 84 2009-10-23 01:26:02Z chenqiang@foresee.com.cn $
 */
package com.foresee.open.sdk.kit;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.InflaterInputStream;

/**
 * <p>
 * ZIP/GZIP压缩与解压缩工具类
 * </p>
 * 
 * @author <a href="mailto:chenqiang@foresee.com.cn">chenqiang</a>
 * @version $LastChangedRevision: 84 $
 */
public class ZipUtil {

    private static final int BUFFER_SIZE = 2048;

    public static final String GZIP = "GZIP";
    public static final String ZIP = "ZIP";

    /**
     * 解压缩BASE64之后的压缩数据
     * 
     * @param encdata BASE64之后的压缩数据
     * @return 解压缩之后的数据
     * @throws IOException IO异常
     */
    public static String decompress(String encdata, boolean codebyB64) {
        if (encdata == null) {
            return null;
        }
        if (codebyB64) {
            return new String(decompress(Base64.decode(encdata)));
        } else {
            return new String(decompress(encdata.getBytes()));
        }
    }

    /**
     * 压缩字符串数据
     * 
     * @param data 字符串数据
     * @return 压缩之后的数据的BASE64编码字符串
     * @throws IOException IO异常
     */
    public static String compress(String data, boolean codebyB64) {
        if (data == null) {
            return null;
        }
        if (codebyB64) {
            return new String(Base64.encode(compress(data.getBytes())));
        } else {
            return new String(compress(data.getBytes()));
        }
    }

    /**
     * 压缩字节数组形式的数据
     * 
     * @param encdata 字节数组形式的数据
     * @return 压缩之后的字节数组
     * @throws IOException IO异常
     */
    public static byte[] compress(byte[] data) {
        ByteArrayOutputStream byteArrayOutputStream = null;
        DeflaterOutputStream deflaterOutputStream = null;
        try {
            byteArrayOutputStream = new ByteArrayOutputStream();
            deflaterOutputStream = new DeflaterOutputStream(byteArrayOutputStream);
            deflaterOutputStream.write(data);
            deflaterOutputStream.close();
            return byteArrayOutputStream.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (deflaterOutputStream != null) {
                try {
                    deflaterOutputStream.close();
                } catch (IOException e) {}
            }
            if (byteArrayOutputStream != null) {
                try {
                    byteArrayOutputStream.close();
                } catch (IOException e) {}
            }
        }
    }

    /**
     * 解压缩字节数组形式的数据
     * 
     * @param encdata 字节数组形式的数据
     * @return 解压缩之后的字节数组
     * @throws IOException IO异常
     */
    public static byte[] decompress(byte[] encdata) {
        if (encdata == null) {
            return null;
        }

        InputStream inputStream = null;
        InflaterInputStream inflaterInputStream = null;
        ByteArrayOutputStream byteArrayOutputStream = null;
        try {
            inputStream = new ByteArrayInputStream(encdata);
            inflaterInputStream = new InflaterInputStream(inputStream);
            byteArrayOutputStream = new ByteArrayOutputStream();

            int count;
            byte[] data = new byte[BUFFER_SIZE];

            while ((count = inflaterInputStream.read(data, 0, BUFFER_SIZE)) != -1) {
                byteArrayOutputStream.write(data, 0, count);
            }
            return byteArrayOutputStream.toByteArray();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {}
            }
            if (inflaterInputStream != null) {
                try {
                    inflaterInputStream.close();
                } catch (IOException e) {}
            }
            if (byteArrayOutputStream != null) {
                try {
                    byteArrayOutputStream.close();
                } catch (IOException e) {}
            }
        }
    }

    /**
     * 使用ZIP压缩算法对字符串进行压缩，并做BASE64编码，使用指定的字符集
     * @param str 待压缩的字符串
     * @param charset 字符集
     * @return 压缩后的字符串
     * @throws IOException IO异常
     */
    public static String zip(String str, String charset) {
        if (null == str || str.length() <= 0) {
            return str;
        }

        if (charset == null) {
            charset = Charset.defaultCharset().name();
        }

        return Base64.encodeToStr(compress(getBytes(str, charset)));
    }

    /**
     * 使用ZIP压缩算法对字符串进行压缩，并做BASE64编码，使用系统默认的字符集
     * @param str 待压缩的字符串
     * @return 压缩后的字符串
     * @throws IOException IO异常
     */
    public static String zip(String str) {
        return zip(str, Charset.defaultCharset().name());
    }

    /**
     * 使用ZIP压缩算法对字节数组进行压缩
     * @param bytes 待压缩的字节数组
     * @return 压缩后的字节数组
     * @throws IOException IO异常
     */
    public static byte[] zip(byte[] bytes) {
        return compress(bytes);
    }

    /**
     * 对BASE64字符串做解压缩，使用指定的字符集
     * @param str 待解压缩的字符串
     * @return 解压缩后的字符串
     * @throws IOException IO异常
     */
    public static String unzip(String str, String charset) {
        if (null == str || str.length() <= 0) {
            return str;
        }

        if (charset == null) {
            charset = Charset.defaultCharset().name();
        }

        return new String(decompress(Base64.decode(str)), getCharset(charset));
    }

    /**
     * 使用GZIP压缩算法对字符串进行压缩，并做BASE64编码，使用指定的字符集
     * @param str 待压缩的字符串
     * @return 压缩后的字符串
     * @throws IOException IO异常
     */
    public static String gzip(String str, String charset) {
        if (null == str || str.length() <= 0) {
            return str;
        }

        return new String(Base64.encode(gzip(getBytes(str, charset))));
    }

    private static byte[] getBytes(String str, String charset) {
        if (null == str || str.length() <= 0) {
            return null;
        }

        return str.getBytes(getCharset(charset));
    }

    private static Charset getCharset(String charset) {
        Charset charSet = null;
        if (charset != null) {
            charSet = Charset.forName(charset);
        }
        if (charSet == null) {
            charSet = Charset.defaultCharset();
        }
        return charSet;
    }

    /**
     * 使用GZIP压缩算法对字符串进行压缩，并做BASE64编码，使用系统默认的字符集
     * @param str 待压缩的字符串
     * @return 压缩后的字符串
     * @throws IOException IO异常
     */
    public static String gzip(String str) {
        return gzip(str, Charset.defaultCharset().name());
    }

    /**
     * 使用GZIP压缩算法对字节数组进行压缩
     * @param bytes 待压缩的字节数组
     * @return 压缩后的字节数组
     * @throws IOException IO异常
     */
    public static byte[] gzip(byte[] bytes) {

        // 创建一个新的 byte 数组输出流
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        GZIPOutputStream gos = null;
        try {
            // 使用默认缓冲区大小创建新的输出流
            gos = new GZIPOutputStream(out);
            // 将 b.length 个字节写入此输出流
            gos.write(bytes);
            gos.close();
            return out.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (gos != null) {
                try {
                    gos.close();
                } catch (IOException e) {}
            }

            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {}
            }
        }
    }

    /**
     * 使用GZIP压缩算法对BASE64字符串解压缩，使用指定的字符集
     * @param str 待解压缩字符串
     * @return 解压缩后的字符串
     * @throws IOException IO异常
     */
    public static String ungzip(String str, String charset) {
        if (null == str || str.length() <= 0) {
            return str;
        }

        if (charset == null) {
            charset = Charset.defaultCharset().name();
        }

        // 对压缩文件数据用BASE64格式进行解码
        byte[] gzipBytes = Base64.decode(str);

        byte[] ungzipBytes = ungzip(gzipBytes);
        // 使用指定的 charsetName，通过解码字节将缓冲区内容转换为字符串
        return new String(ungzipBytes, getCharset(charset));
    }

    /**
     * 使用GZIP压缩算法对BASE64字符串解压缩，使用指定的字符集
     * @param str 待解压缩字符串
     * @return 解压缩后的字符串
     * @throws IOException IO异常
     */
    public static String ungzip(String str) {
        return ungzip(str, Charset.defaultCharset().name());
    }

    /**
     * 字符串的解压
     * @param str 对字符串解压
     * @return 返回解压缩后的字符串
     * @throws IOException IO异常
     */
    public static byte[] ungzip(byte[] bytes) {
        // 创建一个新的 byte 数组输出流
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        // 创建一个 ByteArrayInputStream，使用 buf 作为其缓冲区数组
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        // 使用默认缓冲区大小创建新的输入流
        GZIPInputStream gis = null;

        try {
            gis = new GZIPInputStream(in);
            byte[] buffer = new byte[BUFFER_SIZE];
            int n = 0;
            while ((n = gis.read(buffer)) >= 0) {// 将未压缩数据读入字节数组
                // 将指定 byte 数组中从偏移量 off 开始的 len 个字节写入此 byte数组输出流
                out.write(buffer, 0, n);
            }

            // 使用指定的 charsetName，通过解码字节将缓冲区内容转换为字符串
            return out.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (gis != null) {
                try {
                    gis.close();
                } catch (IOException e) {}
            }
            if (in != null) {
                try {
                    in.close();
                } catch (IOException e) {}
            }
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {}
            }
        }
    }
}