package com.foresee.open.sdk.client;

import com.foresee.open.sdk.client.config.RequestConfig;
import com.foresee.open.sdk.client.ssl.SSLConfig;
import com.foresee.open.sdk.client.ssl.SSLHelper;
import com.foresee.open.sdk.constant.OpenApiConstants;
import com.foresee.open.sdk.constant.OpenApiConstants.EncryptAlgorithm;
import com.foresee.open.sdk.constant.OpenApiConstants.SignAlgorithm;
import com.foresee.open.sdk.exception.OpenApiClientException;
import com.foresee.open.sdk.exception.OpenApiResponseException;
import com.foresee.open.sdk.json.JsonKit;
import com.foresee.open.sdk.kit.DesKit;
import com.foresee.open.sdk.kit.gm.SM4Util;
import okhttp3.*;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * OpenApiClient
 *
 * @param <EncryptAlgorithm>
 * @author chenwenlong@foresee.com.cn
 * @version 1.0
 */
public class OpenHttpClient {

    private static final Map<String, String> debugURLs = new HashMap<String, String>();

    private String appId;

    /**
     * 加密key
     */
    private String encryptKey;

    /**
     * 加密算法
     */
    private EncryptAlgorithm encryptAlgorithm;

    /**
     * 加密算法
     */
    private SignAlgorithm signAlgorithm;

    /**
     * 签名key
     */
    private String signKey;

    private static final MediaType JSON = MediaType.parse("application/json; charset=utf-8");

    private String sessionId = UUID.randomUUID().toString().replace("-", "");

    private static final String ua = "FORESEE OPEN-SDK 2.3.0";

    private OkHttpClient.Builder builder = new OkHttpClient().newBuilder().connectTimeout(10, TimeUnit.SECONDS)
            .readTimeout(180, TimeUnit.SECONDS).writeTimeout(60, TimeUnit.SECONDS)
            .connectionPool(new ConnectionPool(20, 1, TimeUnit.MINUTES)).retryOnConnectionFailure(true);

    private OkHttpClient client;

    private OkHttpClient downloadClient = null;
    //
    //
    //    public OpenHttpClient(String appId, String signKey) {
    //        this.appId = appId;
    //        this.signKey = signKey;
    //    }


    public OpenHttpClient(String appId, String signKey, SignAlgorithm signAlgorithm, String encryptKey,
                          EncryptAlgorithm encryptAlgorithm) {
        this.appId = appId;
        this.encryptKey = encryptKey;
        this.encryptAlgorithm = encryptAlgorithm;
        this.signKey = signKey;
        this.signAlgorithm = signAlgorithm;
        if (SSLConfig.isOpenHttpsRequest()) {
            builder.sslSocketFactory(SSLHelper.getSSLCertifcation());
        }
        if (RequestConfig.isProxyFlag()) {
            builder.proxy(RequestConfig.getProxy());
        }
        if (RequestConfig.isTimeoutFlag()) {
            this.builder.connectTimeout(RequestConfig.getConnectTimeout(), TimeUnit.SECONDS);
            this.builder.readTimeout(RequestConfig.getReadTimeout(), TimeUnit.SECONDS);
            this.builder.writeTimeout(RequestConfig.getWriteTimeout(), TimeUnit.SECONDS);
        }
        this.client = builder.build();
    }

    public String post(OpenApiRequest openApiRequest) {
        Object requestObject = openApiRequest.getRequestObject();
        String requestData;
        if (requestObject instanceof String) {
            requestData = ((String) requestObject);
        } else {
            requestData = JsonKit.toString(requestObject);
        }
        MediaType JSON = MediaType.parse("application/json; charset=utf-8");
        ;
        if (openApiRequest.getContextType() != null) {
            JSON = MediaType.parse(openApiRequest.getContextType().getContextType());
        }
        TreeMap<String, Object> queryMap = genQueryMap(openApiRequest);

        String apiUrl = openApiRequest.getPath();
        String host = openApiRequest.getHost();
        String url = host + apiUrl;

        boolean isDebug = false;
        if (debugURLs.containsKey(apiUrl)) {
            isDebug = true;
            url = debugURLs.get(apiUrl);
        }

        if (!isDebug && encryptKey != null) {
            // SM2 + SM4 特殊处理：使用accessToken中的随机密钥作为加密密钥
            if (this.signAlgorithm == SignAlgorithm.SM2 && this.encryptAlgorithm == EncryptAlgorithm.SM4) {
                String key = openApiRequest.getEncryptionKey() != null ? openApiRequest.getEncryptionKey() : this.encryptKey;
                requestData = SM4Util.encryptDataCBC(key, requestData);
            } else if (this.encryptAlgorithm == EncryptAlgorithm.DES) {
                requestData = DesKit.encrypt(this.encryptKey, requestData);
            } else if (this.encryptAlgorithm == EncryptAlgorithm.SM4) {
                requestData = SM4Util.encryptDataCBC(this.encryptKey, requestData);
            } else {
                throw new OpenApiClientException("未知的加密算法：" + this.encryptAlgorithm);
            }

        }
        String requestId = "未知requestId";
        if (queryMap != null) {
            requestId = queryMap.get(OpenApiConstants.P_REQUEST_ID) + "";
        }

        // SM2 特殊处理：不使用accessToken中的随机密钥签名，而是使用传入的私钥签名
        if (signAlgorithm == SignAlgorithm.SM2) {
            openApiRequest.setSignKey(signKey);
        }

        String queryString = OpenApiHelper.genQueryString(queryMap, openApiRequest.getSignKey(), this.signAlgorithm, requestData);

        url = url + (url.endsWith("?") ? "" : "?") + queryString;

        RequestBody body = RequestBody.create(JSON, requestData);
        Request.Builder builder = new Request.Builder().url(url);
        if (openApiRequest.getRequestHeaders() != null) {
            //设置请求头数据
            for (Map.Entry header : openApiRequest.getRequestHeaders().entrySet()) {
                builder.addHeader(header.getKey() + "", header.getValue() + "");
            }
        }
        Request request = builder.post(body).build();
        try (Response response = client.newCall(request).execute()) {
            String encryptedRespText = response.body().string();
            if (response.code() != 200) {
                throw new OpenApiClientException(url + " 调用接口错误, http错误码" + response.code() + ",请求requestId:" + requestId);
            }
            encryptedRespText = encryptedRespText.trim();
            if (encryptedRespText.startsWith("{")) {
                return encryptedRespText;
            }

            if (!isDebug && encryptKey != null) {
                // SM2 + SM4 特殊处理：使用accessToken中的随机密钥作为解密密钥
                if (this.signAlgorithm == SignAlgorithm.SM2 && this.encryptAlgorithm == EncryptAlgorithm.SM4) {
                    String key = openApiRequest.getEncryptionKey() != null ? openApiRequest.getEncryptionKey() : this.encryptKey;
                    encryptedRespText = SM4Util.decryptDataCBC(key, key, encryptedRespText, true);
                } else if (this.encryptAlgorithm == EncryptAlgorithm.DES) {
                    encryptedRespText = DesKit.decryptAndDecompress(this.encryptKey, encryptedRespText);
                } else if (this.encryptAlgorithm == EncryptAlgorithm.SM4) {
                    encryptedRespText = SM4Util.decryptDataCBC(this.encryptKey, this.encryptKey, encryptedRespText, true);
                } else {
                    throw new OpenApiClientException("未知的加密算法：" + this.encryptAlgorithm + ",请求requestId:" + requestId);
                }
            }

            return encryptedRespText;
        } catch (IOException e) {
            throw new OpenApiClientException("调用接口异常,请求requestId:" + requestId, e);
        }
    }

    private TreeMap<String, Object> genQueryMap(OpenApiRequest openApiRequest) {
        String requestId = UUID.randomUUID().toString().replace("-", "");
        // 外部传进来的queryString
        Map<String, ?> givenQueryMap = openApiRequest.getQueryMap();

        TreeMap<String, Object> queryMap = new TreeMap<>();
        queryMap.put("appId", appId);

        String sessionIdKey = "sessionId";
        String requestIdKey = "requestId";
        String timestampKey = "timestamp";
        String accessTokenKey = "accessToken";

        if (!givenQueryMap.containsKey(sessionIdKey)) {
            queryMap.put(sessionIdKey, this.sessionId);
        }
        if (!givenQueryMap.containsKey(requestIdKey)) {
            queryMap.put(requestIdKey, requestId);
        }
        if (!givenQueryMap.containsKey(timestampKey)) {
            long timestamps = System.currentTimeMillis();
            queryMap.put(timestampKey, String.valueOf(timestamps));
        }
        if (!givenQueryMap.containsKey(accessTokenKey)) {
            queryMap.put(accessTokenKey, openApiRequest.getAccessToken());
        }

        queryMap.putAll(givenQueryMap);
        return queryMap;
    }

    public <T> T execute(OpenApiRequest openApiRequest, Class<T> cls) {
        String respText = this.post(openApiRequest);
        OpenApiResponse<T> entireResponse = JsonKit.fromJson(respText, OpenApiResponse.class);
        if (entireResponse.isFailed()) {
            throw new OpenApiResponseException("请求失败, " + entireResponse.getHead());
        }
        if (cls == Void.class) {
            return null;
        }
        return entireResponse.getBody(cls);
    }

    public String getSignKey() {
        return signKey;
    }

    public OpenHttpClient setSignKey(String signKey) {
        this.signKey = signKey;
        return this;
    }

    public static void addDebugURL(String path, String debugURL) {
        debugURLs.put(path, debugURL);
    }

    public static void removeDebugURL(String path) {
        debugURLs.remove(path);
    }

    public String download(String url, int timeoutMills) {

        OkHttpClient client = getDownloadHttpClient(timeoutMills);

        Request request = new Request.Builder().url(url).addHeader("User-Agent", ua).get().build();
        try (Response response = client.newCall(request).execute()) {
            String encryptedRespText = response.body().string();
            if (response.code() != 200) {
                throw new OpenApiClientException(url + " 下载API调用结果错误, http错误码" + response.code());
            }
            encryptedRespText = encryptedRespText.trim();

            return encryptedRespText;
        } catch (IOException e) {
            throw new OpenApiClientException("下载API调用结果错误：" + url, e);
        }
    }

    private synchronized OkHttpClient getDownloadHttpClient(int timeoutMills) {

        if (downloadClient == null) {
            downloadClient = new OkHttpClient().newBuilder().connectTimeout(10, TimeUnit.SECONDS)
                    .readTimeout(timeoutMills, TimeUnit.MILLISECONDS).writeTimeout(1, TimeUnit.MINUTES)
                    .connectionPool(new ConnectionPool(20, 1, TimeUnit.MINUTES)).retryOnConnectionFailure(true).build();
        }

        return downloadClient;
    }

}
