package com.foresee.open.sdk.client;

import com.foresee.open.sdk.constant.OpenApiConstants.SignAlgorithm;
import com.foresee.open.sdk.exception.OpenApiClientException;
import com.foresee.open.sdk.kit.DigestKit;
import com.foresee.open.sdk.kit.gm.SM2Util;
import com.foresee.open.sdk.kit.gm.SM3Util;
import com.google.common.base.Strings;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;

/**
 * ClientHelper
 *
 * @author chenwenlong@foresee.com.cn
 * @version 1.0
 */
public abstract class OpenApiHelper {


    /**
     * 1、对所有 URL 参数按照参数名称的 ASCII 码表的顺序排序，然后拼接在一
     * 起，如：
     * URL 参数拼接  =  “accessToken=<值>&appId=<值>&requestId=<值>
     * &sessionId=<值>&timestamp=<值>”
     * <p>
     * 2、对 URL 参数拼接、请求体 JSON 报文、签名密钥进行连接：
     * 待签名字符串  = URL 参数拼接  +  “&”+  请求体 JSON 报文  +  “{”+  签
     * 名密钥  +“}”
     * 5、对待签名字符串做 HASH：
     * 签名值  =  签名算法(待签名字符串)
     * MD5 算法：  签名值  = UpperCase(MD5(待签名字符串)))
     * SHA-256 算法：  签名值  = UpperCase(MD5(SHA-256(待签名字符串))
     * <p>
     * 响应报文只对响应体的 JSON 报文进行签名：
     * 签名值  =  签名算法(响应体 JSON 报文  +  “{”+  签名密钥  +“}”)
     * <p>
     * （开放平台统一提供 Java 版签名 SDK）
     *
     * @param jsonData
     * @return
     */

    /**
     * 私钥-公钥
     */
    private static final ConcurrentHashMap<String, String> keyPair = new ConcurrentHashMap<>(20);


    public static boolean validateSign(String queryString, String signKey, SignAlgorithm signMethod, String jsonData) {
        return doValidateSign(queryString, signKey, signMethod, jsonData);
    }

    private static boolean doValidateSign(String queryString, String signKey, SignAlgorithm signMethod, String jsonData) {
        Map<String, String> queryPairs = queryStringToMap(queryString);
        String expectedSign = queryPairs.remove("sign");
        Map<String, ?> queryMap = new TreeMap<>(queryPairs);
        String query = getQueryString(queryMap);
        String actualSign = calSign(query, signKey, signMethod, jsonData);
        return actualSign.equals(expectedSign);
    }

    public static Map<String, String> queryStringToMap(String queryString) {
        try {
            if (Strings.isNullOrEmpty(queryString)) {
                return Collections.emptyMap();
            }
            Map<String, String> queryPairs = new LinkedHashMap<>();
            String[] pairs = queryString.split("&");
            for (String pair : pairs) {
                int idx = pair.indexOf("=");
                queryPairs.put(URLDecoder.decode(pair.substring(0, idx), "UTF-8"),
                        URLDecoder.decode(pair.substring(idx + 1), "UTF-8"));
            }
            return queryPairs;
        } catch (UnsupportedEncodingException e) {
            throw new OpenApiClientException(queryString + "解码失败", e);
        }
    }

    public static String genQueryString(Map<String, ?> map, String signKey, SignAlgorithm signAlgorithm, String jsonData) {

        Map<String, Object> queryMap = new TreeMap<>(map);
        String queryString = getQueryString(queryMap);
        String signedText = calSign(queryString, signKey, signAlgorithm, jsonData);
        queryString = new StringBuilder().append(queryString).append("sign=").append(signedText).toString();
        return queryString;
    }

    private static String calSign(String queryString, String signKey, SignAlgorithm signAlgorithm, String jsonData) {

        StringBuilder sb = new StringBuilder(queryString);

        // SM2 特殊处理
        if (signAlgorithm == SignAlgorithm.SM2) {
            String publicKey = keyPair.get(signKey);
            if (publicKey == null) {
                publicKey = SM2Util.generatePublicKey(signKey);
                keyPair.put(signKey, publicKey);
            }
            // 通过私钥获取到公钥，加入签名参数
            String messageText = sb.append(jsonData).append("{").append(publicKey).append("}").toString();
            return SM2Util.sign(signKey, messageText);
        }

        String messageText = sb.append(jsonData).append("{").append(signKey).append("}").toString();
        // 签名数据
        if (signAlgorithm == SignAlgorithm.SHA256) {
            byte[] bytes = DigestKit.sha256(messageText);
            return DigestKit.md5Hex(bytes).toUpperCase();
        } else if (signAlgorithm == SignAlgorithm.MD5) {
            return DigestKit.md5Hex(messageText.getBytes()).toUpperCase();
        } else if (signAlgorithm == SignAlgorithm.SM3) {
            return DigestKit.md5Hex(SM3Util.byteArrayToHexString(messageText.getBytes())).toUpperCase();
        } else {
            throw new RuntimeException("未知的签名算法：" + signAlgorithm);
        }
    }

    private static String getQueryString(Map<String, ?> queryMap) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, ?> entry : queryMap.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            String valueStr = value == null ? "" : value.toString();

            if (valueStr.trim().equals("")) {
                continue;
            }
            sb.append(key).append("=").append(valueStr).append("&");
        }
        return sb.toString();
    }

    public static URI getUri(String url) {
        try {
            return new URI(url);
        } catch (URISyntaxException e) {
            throw new OpenApiClientException(url + "不是一个完整的url", e);
        }
    }
}
