package com.foresee.open.sdk.client;

import java.lang.reflect.Proxy;
import java.util.concurrent.Callable;
import com.foresee.open.sdk.api.BaseOpenApi;
import com.foresee.open.sdk.api.OpenApiInvocationHandler;
import com.foresee.open.sdk.api.OpenApiService;
import com.foresee.open.sdk.api.ServiceInfo;
import com.foresee.open.sdk.constant.OpenApiConstants;
import com.foresee.open.sdk.constant.OpenApiConstants.EncryptAlgorithm;
import com.foresee.open.sdk.constant.OpenApiConstants.SignAlgorithm;
import com.foresee.open.sdk.exception.OpenApiCreationException;
import com.foresee.open.sdk.exception.OpenApiFactoryException;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

/**
 * OpenApiFactory
 *
 * @author chenwenlong@foresee.com.cn
 * @version 1.0
 */
public class OpenApiFactory {

    private String                          appId;

    private String                          signKey;

    private OpenHttpClient                  openHttpClient;

    private Cache<Class<?>, Object>         cache              = CacheBuilder.newBuilder().maximumSize(1000).build();


    /**
     * Create a new instance of OpenApiFactory.
     * 
     * @param appId 应用ID
     * @param signKey 签名密钥（默认使用SHA-256签名算法）
     */
    public OpenApiFactory(String appId, String signKey) {
        this(appId, signKey, OpenApiConstants.DEFAULT_SIGN_ALGORITHM, null, OpenApiConstants.DEFAULT_ENCRYPT_ALGORITHM);
    }

    /**
     * Create a new instance of OpenApiFactory.
     * @param appId 应用ID
     * @param signKey 签名密钥（默认使用SHA-256签名算法）
     * @param encryptKey 加密密钥（默认使用DES加密算法）
     */
    public OpenApiFactory(String appId, String signKey, String encryptKey) {
        this(appId, signKey, OpenApiConstants.DEFAULT_SIGN_ALGORITHM, encryptKey, OpenApiConstants.DEFAULT_ENCRYPT_ALGORITHM);
    }

    /**
     * Create a new instance of OpenApiFactory.
     * @param appId 应用ID
     * @param signKey 签名密钥
     * @param signAlgorithm 签名算法
     * @param encryptKey 加密密钥
     * @param encryptAlgorithm 加密算法
     */
    public OpenApiFactory(String appId, String signKey, SignAlgorithm signAlgorithm, String encryptKey,
            EncryptAlgorithm encryptAlgorithm) {
        this.appId = appId;
        this.signKey = signKey;
        openHttpClient = new OpenHttpClient(appId, signKey, signAlgorithm, encryptKey, encryptAlgorithm);
    }

    public <T extends BaseOpenApi> T create(final Class<T> klass) {
        try {
            return (T) cache.get(klass, new Callable<Object>() {

                @Override
                public Object call() throws Exception {
                    return doCreate(klass);
                }
            });
        } catch (Exception e) {
            throw new OpenApiCreationException(klass + " api创建失败", e);
        }

    }

    private <T extends BaseOpenApi> T doCreate(Class<T> klass) throws Exception {
        // 检查klass是否符合OpenApi的定义
        checkClass(klass);

        ServiceInfo serviceInfo = parseServiceInfo(klass);
        // 遍历文件夹生成api
        T t = generate(klass, serviceInfo);

        return t;
    }

    private <T extends BaseOpenApi> ServiceInfo parseServiceInfo(Class<T> klass) {
        ServiceInfo serviceInfo = new ServiceInfo(null, null);
        OpenApiService openApiService = klass.getAnnotation(OpenApiService.class);
        if (openApiService == null && klass.getInterfaces().length > 0) {
            for (Class<?> clazz : klass.getInterfaces()) {
                openApiService = clazz.getAnnotation(OpenApiService.class);
                if (openApiService != null) {
                    break;
                }
            }

        }

        if (openApiService != null) {
            serviceInfo.setAppId(openApiService.appId());
            serviceInfo.setService(openApiService.name());
        }

        if (serviceInfo.getAppId() == null) {
            throw new RuntimeException(String.format("接口[%s]未配置appId。", klass.getName()));
        }
        if (serviceInfo.getService() == null) {
            throw new RuntimeException(String.format("接口[%s]未配置服务名。", klass.getName()));
        }

        return serviceInfo;
    }

    private <T extends BaseOpenApi> T generate(Class<T> klass, ServiceInfo serviceInfo) {
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        return (T) Proxy.newProxyInstance(classLoader, new Class[] { klass }, new OpenApiInvocationHandler(this, serviceInfo));
    }

    private <T extends BaseOpenApi> void checkClass(Class<T> klass) {
        if (!klass.isInterface()) {
            throw new OpenApiFactoryException(klass + "不是一个接口");
        }

        if (!BaseOpenApi.class.isAssignableFrom(klass)) {
            throw new OpenApiFactoryException(klass + "没有继承" + BaseOpenApi.class);
        }
    }

    public String getServiceInfo() {
        return appId;
    }

    public String getSignKey() {
        return signKey;
    }

    public String getAppId() {
        return appId;
    }
    
    public OpenHttpClient getOpenHttpClient() {
        return openHttpClient;
    }
    
    public OpenApiClient getOpenApiClient() {
        return new OpenApiClient(this);
    }

}
