package com.foresee.open.sdk.client;

import com.foresee.open.sdk.api.AccessTokenHelper;
import com.foresee.open.sdk.api.ExtMapConfig;
import com.foresee.open.sdk.auth.AuthHost;
import com.foresee.open.sdk.auth.GatewayConfig;
import com.foresee.open.sdk.auth.vo.AccessToken;
import com.foresee.open.sdk.constant.ExceptionConstants;
import com.foresee.open.sdk.constant.OpenApiConstants;
import com.foresee.open.sdk.constant.OpenApiConstants.EncryptAlgorithm;
import com.foresee.open.sdk.constant.OpenApiConstants.SignAlgorithm;
import com.foresee.open.sdk.exception.OpenApiResponseException;
import com.foresee.open.sdk.json.JsonKit;
import com.foresee.open.sdk.kit.DesKit;
import com.foresee.open.sdk.kit.ZipUtil;
import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.Proxy;
import java.net.URI;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * OpenApiClient
 *
 * @author chenwenlong@foresee.com.cn
 * @version 1.0
 */
public class OpenApiClient {

    /**
     * 日志记录
     */
    private static final Logger logger = LoggerFactory.getLogger(OpenApiClient.class);

    /*
     * 异步查询超时时间，默认30分钟
     */
    int asynchQueryTimeout = 30 * 60 * 1000;

    private OpenHttpClient openHttpClient;

    private OpenApiFactory openApiFactory;

    /**
     * 是否启用扩展处理，默认启用
     */
    private boolean enabledExtHandling = true;

    private AtomicInteger retryCount = new AtomicInteger(1);


    /**
     * 创建开放API调用客户端
     *
     * @param appId   消费方应用ID
     * @param signKey 签名密钥（默认使用SHA-256签名算法）
     */
    public OpenApiClient(String appId, String signKey) {
        this.openApiFactory = new OpenApiFactory(appId, signKey);
        this.openHttpClient = this.openApiFactory.getOpenHttpClient();
    }

    /**
     * 创建开放API调用客户端
     *
     * @param appId   消费方应用ID
     * @param signKey 签名密钥（默认使用SHA-256签名算法）
     */
    public OpenApiClient(String appId, String signKey, Proxy proxy) {
        GatewayConfig.setProxy(proxy);
        this.openApiFactory = new OpenApiFactory(appId, signKey);
        this.openHttpClient = this.openApiFactory.getOpenHttpClient();

    }

    /**
     * 创建开放API调用客户端
     *
     * @param appId      应用ID
     * @param signKey    签名密钥（默认使用SHA-256签名算法）
     * @param encryptKey 加密密钥（默认使用DES加密算法）
     */
    public OpenApiClient(String appId, String signKey, String encryptKey) {
        this.openApiFactory = new OpenApiFactory(appId, signKey, OpenApiConstants.DEFAULT_SIGN_ALGORITHM, encryptKey,
                OpenApiConstants.DEFAULT_ENCRYPT_ALGORITHM);
        this.openHttpClient = this.openApiFactory.getOpenHttpClient();
    }

    /**
     * 创建开放API调用客户端
     *
     * @param appId         消费方应用ID
     * @param signKey       签名密钥
     * @param signAlgorithm 签名算法：NONE、MD5, SHA256,SM3
     * @param encryptKey    加密密钥
     */
    public OpenApiClient(String appId, String signKey, SignAlgorithm signAlgorithm) {
        this.openApiFactory = new OpenApiFactory(appId, signKey, signAlgorithm, null, OpenApiConstants.DEFAULT_ENCRYPT_ALGORITHM);
        this.openHttpClient = this.openApiFactory.getOpenHttpClient();
    }

    /**
     * 创建开放API调用客户端
     *
     * @param appId            消费方应用ID
     * @param signKey          签名密钥
     * @param signAlgorithm    签名算法：NONE、MD5, SHA256,SM3
     * @param encryptKey       加密密钥
     * @param encryptAlgorithm 加密算法：NONE、DES,SM4
     */
    public OpenApiClient(String appId, String signKey, SignAlgorithm signAlgorithm, String encryptKey,
                         EncryptAlgorithm encryptAlgorithm) {
        this.openApiFactory = new OpenApiFactory(appId, signKey, signAlgorithm, encryptKey, encryptAlgorithm);
        this.openHttpClient = this.openApiFactory.getOpenHttpClient();
    }

    /**
     * 从OpenApiFactory创建API调用客户端
     *
     * @param openApiFactory OpenApiFactory
     */
    protected OpenApiClient(OpenApiFactory openApiFactory) {
        this.openApiFactory = openApiFactory;
        this.openHttpClient = this.openApiFactory.getOpenHttpClient();
    }

    /**
     * 请求公开级别接口
     *
     * @param url    自己拼接好的url, 可以不用带sign参数
     * @param object pojo或者map
     * @return
     */
    public String requestPublic(String url, Object object) {
        String body = JsonKit.toString(object);
        return this.requestPublic(url, body);
    }

    /**
     * 请求公开级别接口
     *
     * @param url  自己拼接好的url, 可以不用带sign参数
     * @param body json后的字符串
     * @return
     */
    public String requestPublic(String url, String body) {
        return requestPublic(url, body, null);
    }

    /**
     * 请求公开级别接口
     *
     * @param url                    自己拼接好的url, 可以不用带sign参数
     * @param body                   json后的字符串
     * @param requestHeaderMap:请求头数据
     * @return
     */
    public String requestPublic(String url, String body, Map<String, Object> requestHeaderMap) {
        OpenApiRequest openApiRequest = urlToApiRequest(url).setRequestObject(body).setSignKey(this.openHttpClient.getSignKey());
        setRequestHeaders(openApiRequest, requestHeaderMap);
        return this.openHttpClient.post(openApiRequest);
    }

    public String requestApp(String url, String body, String providerAppId) {
        return this.requestApp(url, body, providerAppId, OpenApiConstants.RequestContextType.JSON, null);
    }

    /**
     * @param url                    自己拼接好的url, 可以不用带sign参数
     * @param body                   请求报文数据
     * @param providerAppId          服务方appId
     * @param requestHeaderMap:请求头数据
     * @return
     */
    public String requestApp(String url, String body, String providerAppId, Map<String, Object> requestHeaderMap) {
        return this.requestApp(url, body, providerAppId, OpenApiConstants.RequestContextType.JSON, requestHeaderMap);
    }

    /**
     * @param url           自己拼接好的url, 可以不用带sign参数
     * @param object        pojo或者map
     * @param providerAppId 服务方appId
     * @return
     */
    public String requestApp(String url, Object object, String providerAppId) {
        String body = JsonKit.toString(object);
        return this.requestApp(url, body, providerAppId, OpenApiConstants.RequestContextType.JSON, null);
    }

    /**
     * @param url                    自己拼接好的url, 可以不用带sign参数
     * @param body                   json后的字符串
     * @param providerAppId          服务方appId
     * @param requestHeaderMap:请求头数据
     * @return
     */
    public String requestApp(String url, String body, String providerAppId, OpenApiConstants.RequestContextType requestContextType,
                             Map<String, Object> requestHeaderMap) {

        if (Strings.isNullOrEmpty(AuthHost.getAuthHost())) {
            OpenApiRequest openApiRequest = urlToApiRequest(url);
            AuthHost.setAuthHost(openApiRequest.getHost());
        }
        AccessToken accessToken = AccessTokenHelper.getAppAccessToken(openApiFactory, providerAppId);

        String responseText = requestByAccessToken(url, body, accessToken, requestContextType, requestHeaderMap);

        //判断token是否已失效
        boolean accessTokenFlag = invalidAccessToken(url, responseText);
        if (accessTokenFlag) {//accessToken已失效,重试调用接口
            final int maxRetryCount=GatewayConfig.getRetryTotalCount();
            if (isMaxRetryCount(maxRetryCount)) {
                logger.error("call[{}] get access token api failure retry count configuration maximum reached [{}]",url,maxRetryCount);
                throw new OpenApiResponseException("100100000003","调用获取AccessToken重试失败次数已达配置最大值",null);
            }
            return requestApp(url, body, providerAppId, requestContextType, requestHeaderMap);

        }
        //重置失败次数
        restRetryCount();

        // 如果不启用扩展处理直接返回报文
        if (!enabledExtHandling) {
            return responseText;
        }

        long startTime = System.currentTimeMillis();

        while (true) {
            ExtMapConfig config = parseExtMapConfig(responseText);
            if (config == null) {
                return responseText;
            }

            if (config.getDownloadUrl() != null) {
                String downloadUrl = config.getDownloadUrl();
                if (!downloadUrl.startsWith("http")) {
                    downloadUrl = DesKit.decrypt("04810679", config.getDownloadUrl());
                }
                logger.info("Downloading from file: {}", downloadUrl);

                long time = System.currentTimeMillis();
                responseText = openHttpClient.download(downloadUrl, asynchQueryTimeout);
                time = System.currentTimeMillis() - time;
                int fileSize = responseText == null ? 0 : responseText.length();
                logger.info("Download finished: {}, use time(ms): {}, file size: {}.", downloadUrl, time, fileSize);

                if (config.getContentEncoding() != null) {
                    try {
                        String decompressResponseBody = ZipUtil.decompress(responseText, true);
                        config.getResponse().setBody(decompressResponseBody);
                        responseText = JsonKit.toString(config.getResponse());
                    } catch (Exception e) {
                        throw new OpenApiResponseException("100100000006", "解压缩报文出错：" + e.getMessage(), null);
                    }
                }

                return responseText;
            }

            if (config.getAsynchQuery() != null) {

                if (config.getAsynchQuery().getWaitTime() > 0) {
                    try {
                        logger.info("Waiting {} seconds for next asynchronous result: {}", config.getAsynchQuery().getWaitTime(),
                                JsonKit.toString(config.getAsynchQuery()));
                        TimeUnit.SECONDS.sleep(config.getAsynchQuery().getWaitTime());
                    } catch (InterruptedException e) {
                        throw new OpenApiResponseException("100100000001", "等待异步查询错误，线程被中断：" + e.getMessage(), null);
                    }
                }

                if (System.currentTimeMillis() - startTime > asynchQueryTimeout) {
                    throw new OpenApiResponseException("100100000002",
                            "等待异步查询超时，总等待时长(秒)：" + String.valueOf((System.currentTimeMillis() - startTime) / 1000), null);
                }

                // 根据异常查询配置设置API路径及请求报文，重新请求
                url = config.getAsynchQuery().getApiUrl();
                body = config.getAsynchQuery().getRequestBody();

                logger.info("Invoking for asynchronous result: {}", JsonKit.toString(config.getAsynchQuery()));

                responseText = requestByAccessToken(url, body, accessToken, OpenApiConstants.DEFAULT_CONTEXT_TYPE, requestHeaderMap);

                continue;
            }

            if (config.getContentEncoding() != null && config.getResponse().getBody() instanceof String) {
                try {
                    logger.info("Unzipping data: {}", config.getResponse().getBody().toString());
                    String decompressResponseBody = ZipUtil.decompress((String) config.getResponse().getBody(), true);
                    config.getResponse().setBody(decompressResponseBody);
                    responseText = JsonKit.toString(config.getResponse());

                    logger.info("Unzipped data: {}", decompressResponseBody);
                } catch (Exception e) {
                    throw new OpenApiResponseException("100100000003", "解压缩报文出错：" + e.getMessage(), null);
                }
            }

            return responseText;
        }

    }

    /**
     * 判断accessToken是否已失效，如果已失效，再重新调用一次，如果未失效，返回false
     *
     * @param responseText
     */
    private boolean invalidAccessToken(String url, String responseText) {
        OpenApiResponse response = JsonKit.fromJson(responseText, OpenApiResponse.class);
        if (response == null) {
            logger.warn("request url:[{}] get response is null", url);
            return false;
        }
        if (response.isFailed()) {
            if (ExceptionConstants.ACCESS_TOKEN_INVALID.equals(response.getHead().getErrorCode())
                    || ExceptionConstants.ACCESS_TOKEN_INVALID_MSG.equals(response.getHead().getErrorMsg())) {
                logger.error("request url:[{}] access token invalid,retry the request count:[{}]", url,retryCount.intValue());
                return true;
            }
        }
        return false;
    }

    /**
     * XMl方式请求后台应用
     *
     * @param url           自己拼接好的url, 可以不用带sign参数
     * @param body          json后的字符串
     * @param providerAppId 服务方appId
     * @return
     */
    public String requestXmlApp(String url, String body, String providerAppId) {
        return requestXmlApp(url, body, providerAppId, null);
    }

    /**
     * XMl方式请求后台应用
     *
     * @param url                    自己拼接好的url, 可以不用带sign参数
     * @param body                   json后的字符串
     * @param providerAppId          服务方appId
     * @param requestHeaderMap:请求头数据
     * @return
     */
    public String requestXmlApp(String url, String body, String providerAppId, Map<String, Object> requestHeaderMap) {
        return this.requestApp(url, body, providerAppId, OpenApiConstants.RequestContextType.XML, requestHeaderMap);
    }

    protected String requestByAccessToken(String url, String body, AccessToken accessToken,
                                          OpenApiConstants.RequestContextType requestContextType) {
        return requestByAccessToken(url, body, accessToken, requestContextType, null);
    }

    protected String requestByAccessToken(String url, String body, AccessToken accessToken,
                                          OpenApiConstants.RequestContextType requestContextType, Map<String, Object> requestHeaderMap) {
        OpenApiRequest openApiRequest = urlToApiRequest(url).setAccessToken(accessToken.getAccessToken()).setRequestObject(body)
                .setContextType(requestContextType).setSignKey(accessToken.getRandomKey()).setEncryptionKey(accessToken.getRandomKey());

        if (!GatewayConfig.isIgnoreProviderURL() && !Strings.isNullOrEmpty(accessToken.getProviderAppURL())) {
            openApiRequest.setHost(accessToken.getProviderAppURL());
        }
        //设置请求头数据
        setRequestHeaders(openApiRequest, requestHeaderMap);
        return this.openHttpClient.post(openApiRequest);
    }

    private void setRequestHeaders(OpenApiRequest openApiRequest, Map<String, Object> requestHeaderMap) {
        //设置请求头
        if (requestHeaderMap != null && !requestHeaderMap.isEmpty()) {
            openApiRequest.getRequestHeaders().putAll(requestHeaderMap);
        }
    }

    private ExtMapConfig parseExtMapConfig(String responseText) {
        if (responseText == null) {
            return null;
        }

        // 如果不包含关键字就不解析报文，避免解析大报文的消耗
        if (!responseText.contains("asynchQuery") && !responseText.contains("contentEncoding") && !responseText.contains("downloadUrl")) {
            return null;
        }

        OpenApiResponse response = JsonKit.fromJson(responseText, OpenApiResponse.class);
        if (response == null || response.getExtMap() == null) {
            return null;
        }

        logger.debug("ExtMap: {}", JsonKit.toString(response.getExtMap()));

        ExtMapConfig config = JsonKit.fromJson(JsonKit.toString(response.getExtMap()), ExtMapConfig.class);

        if (config.getAsynchQuery() != null) {
            if (Strings.isNullOrEmpty(config.getAsynchQuery().getApiUrl())
                    || Strings.isNullOrEmpty(config.getAsynchQuery().getRequestBody())) {
                throw new OpenApiResponseException("100100000004", "异步查询配置错误：" + response.getExtMap().get("asynchQuery"), null);
            }
        }

        if (config.getContentEncoding() != null && !"zip".equalsIgnoreCase(config.getContentEncoding())) {
            throw new OpenApiResponseException("100100000005", "不支持的压缩算法：" + config.getContentEncoding(), null);
        }

        response.getExtMap().remove("asynchQuery");
        response.getExtMap().remove("contentEncoding");
        response.getExtMap().remove("downloadUrl");

        config.setResponse(response);

        return config;
    }

    /**
     * @param url       自己拼接好的url, 可以不用带sign参数
     * @param object    pojo或者map
     * @param userToken 用户登录之后拿到的token
     * @return
     */
    public String requestUser(String url, Object object, String providerAppId, String userToken) {
        String body = JsonKit.toString(object);
        return this.requestUser(url, body, providerAppId, userToken);
    }

    /**
     * @param url                    自己拼接好的url, 可以不用带sign参数
     * @param body                   json后的字符串
     * @param userToken              用户登录之后拿到的token
     * @param requestHeaderMap:请求头数据
     * @return
     */
    public String requestUser(String url, String body, String providerAppId, String userToken, Map<String, Object> requestHeaderMap) {
        if (Strings.isNullOrEmpty(AuthHost.getAuthHost())) {
            OpenApiRequest openApiRequest = urlToApiRequest(url);
            AuthHost.setAuthHost(openApiRequest.getHost());
        }
        AccessToken accessToken = AccessTokenHelper.getUserAccessToken(openApiFactory, providerAppId, userToken);

        return requestByAccessToken(url, body, accessToken, OpenApiConstants.DEFAULT_CONTEXT_TYPE, requestHeaderMap);
    }

    /**
     * @param url       自己拼接好的url, 可以不用带sign参数
     * @param body      json后的字符串
     * @param userToken 用户登录之后拿到的token
     * @return
     */
    public String requestUser(String url, String body, String providerAppId, String userToken) {
        return requestUser(url, body, providerAppId, userToken, null);
    }

    private OpenApiRequest urlToApiRequest(String url) {
        URI uri = OpenApiHelper.getUri(url);
        String path = uri.getPath();
        String baseUrl;
        if (uri.getScheme() != null && uri.getAuthority() != null) {
            baseUrl = uri.getScheme() + "://" + uri.getAuthority();
        } else {
            baseUrl = GatewayConfig.getAddress();
        }

        String query = uri.getQuery();

        Map<String, String> map = OpenApiHelper.queryStringToMap(query);

        return new OpenApiRequest(path).setHost(baseUrl).setQueryMap(map);
    }

    public boolean isEnabledExtHandling() {
        return enabledExtHandling;
    }

    public void setEnabledExtHandling(boolean enabledExtHandling) {
        this.enabledExtHandling = enabledExtHandling;
    }

    /**
     * 获取异步查询超时时间（毫秒）
     */
    public int getAsynchQueryTimeout() {
        return asynchQueryTimeout;
    }

    /**
     * 设置异步查询超时时间（毫秒）
     */
    public void setAsynchQueryTimeout(int asynchQueryTimeout) {
        this.asynchQueryTimeout = asynchQueryTimeout;
    }

    private void restRetryCount() {
        retryCount.set(1);
    }

    /**
     * 判断是不是已达最大失败次数
     *
     * @return
     */
    private boolean isMaxRetryCount(int maxRetryCount) {
        return retryCount.incrementAndGet()> maxRetryCount;
    }
}
