package com.foresee.open.sdk.api;

import com.foresee.open.page.vo.PageResult;
import com.foresee.open.sdk.auth.GatewayConfig;
import com.foresee.open.sdk.auth.vo.AccessToken;
import com.foresee.open.sdk.client.OpenApiFactory;
import com.foresee.open.sdk.client.OpenApiRequest;
import com.foresee.open.sdk.client.OpenApiResponse;
import com.foresee.open.sdk.constant.ExceptionConstants;
import com.foresee.open.sdk.exception.OpenApiClientException;
import com.foresee.open.sdk.exception.OpenApiCreationException;
import com.foresee.open.sdk.exception.OpenApiResponseException;
import com.foresee.open.sdk.json.Jackson;
import com.foresee.open.sdk.json.JsonKit;
import com.foresee.open.sdk.kit.ClassKit;
import com.google.common.base.Strings;
import sun.reflect.generics.reflectiveObjects.ParameterizedTypeImpl;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * OpenApiInvocationHandler
 *
 * @author chenwenlong@foresee.com.cn
 * @version 1.0
 */
public class OpenApiInvocationHandler implements InvocationHandler {

    private ServiceInfo serviceInfo;

    private OpenApiFactory openApiFactory;


    public OpenApiInvocationHandler(OpenApiFactory openApiFactory, ServiceInfo serviceInfo) {
        this.serviceInfo = serviceInfo;
        this.openApiFactory = openApiFactory;
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        return invokeOpenMethod(proxy, method, args, false);
    }

    private Object invokeOpenMethod(Object proxy, Method method, Object[] args, boolean isRepeat) throws Throwable {
        Object body = assembleBody(method, args);
        OpenApiVO openApiVO = getOpenApi(method);
        if (GatewayConfig.isSkipGateway()) {
            openApiVO.setLevel(OpenApiLevel.PUBLIC);
            String path = openApiVO.getPath();
            String[] paths = path.split("/");
            openApiVO.setPath("/" + paths[paths.length - 2] + "/" + paths[paths.length - 1]);
        }

        OpenApiRequest request = new OpenApiRequest(openApiVO.getPath());
        if (body != null) {
            request.setRequestObject(body);
        }
        switch (openApiVO.getLevel()) {
            case PUBLIC:
                initializePublic(method, request);
                break;
            case USER:
                // 根据userToken拿userAccessToken, userAccessToken
                //
                String userToken = getUserToken(body); // (String) body.get("userToken");
                initializeUser(request, userToken);
                break;
            case APP:
                // 根据serviceInfo.appId拿appAccessToken
                initializeApp(request);
                break;
        }

        if (Strings.isNullOrEmpty(request.getHost())) {
            if (!Strings.isNullOrEmpty(GatewayConfig.getAddress())) {
                request.setHost(GatewayConfig.getAddress());
            } else {
                throw new OpenApiClientException("API网关地址还没有设置，请调用GatewayConfig.setAddress方法设置。");
            }
        }
        String respText = openApiFactory.getOpenHttpClient().post(request);
        Type genericReturnType = method.getGenericReturnType();
        Class<?> returnType = method.getReturnType();


        // TODO 解密
        OpenApiResponse entireResponse = JsonKit.fromJson(respText, OpenApiResponse.class);

        // 判断是否请求失败，失败原因是否为accessToken失效，失效重发,最多重发一次,
        if (entireResponse.isFailed() && entireResponse.getHead().getErrorCode().equalsIgnoreCase(ExceptionConstants.ACCESS_TOKEN_INVALID)) {
            if (!isRepeat) {
                AccessTokenHelper.cleanCache();
                return invokeOpenMethod(proxy, method, args, true);
            }
        }


        /** 分页也可能是泛型，所以要增加判断依据是否为分页类型 */
        if (returnType.isAssignableFrom(PageResult.class) || !(genericReturnType instanceof ParameterizedTypeImpl)) {
            // 不是泛型

            if (returnType.isAssignableFrom(OpenApiResponse.class)) {
                return entireResponse;
            }

            if (entireResponse.isFailed()) {
                OpenApiResponse.Head head = entireResponse.getHead();
                throw new OpenApiResponseException(head.getErrorCode(), head.getErrorMsg(), head.getRequestId());
            }
            if (returnType.isAssignableFrom(void.class)) {
                return null;
            }

            return entireResponse.getBody(returnType);
        }
        ParameterizedTypeImpl genericReturnTypeImpl = (ParameterizedTypeImpl) genericReturnType;
        Type type = genericReturnTypeImpl.getActualTypeArguments()[0];
        Object javaBody = entireResponse.getBody();
        // 如果返回类型是OpenApiResponse或OpenApiResponse的子类，则直接返回
        if (returnType.isAssignableFrom(OpenApiResponse.class)) {
            // Object o = JsonKit.fromJson(JsonKit.toString(javaBody), (Class<?>) type);
            Object o = Jackson.fromJson(JsonKit.toString(javaBody), type);
            entireResponse.setBody(o);
            return entireResponse;
        } else {
            Object o = JsonKit.fromJson(JsonKit.toString(javaBody), returnType, (Class<?>) type);
            return o;
        }
    }

    private String getUserToken(Object body) {
        if (body instanceof Map) {
            return (String) ((Map) body).get("userToken");
        }
        throw new IllegalStateException("userToken没有传入");
    }

    private void initializeApp(OpenApiRequest request) {
        AccessToken appAccessToken = AccessTokenHelper.getAppAccessToken(this.openApiFactory, serviceInfo.getAppId());
        populateRequestByAccessToken(request, appAccessToken);
    }

    private void initializeUser(OpenApiRequest request, String userToken) {
        AccessToken userAccessToken = AccessTokenHelper.getUserAccessToken(openApiFactory, serviceInfo.getAppId(), userToken);
        populateRequestByAccessToken(request, userAccessToken);
    }

    private void populateRequestByAccessToken(OpenApiRequest request, AccessToken userAccessToken) {
        request.setSignKey(userAccessToken.getRandomKey()).setAccessToken(userAccessToken.getAccessToken())
                .setHost(userAccessToken.getProviderAppURL()).setEncryptionKey(userAccessToken.getRandomKey());
    }

    private Object assembleBody(Method method, Object[] args) {
        ApiParamVO[] apiParams = lookupParameterNames(method);
        Class<?>[] parameterTypes = method.getParameterTypes();

        // 参数只有一个,并且是复合类型
        if (apiParams.length == 1 && !isSimpleType(parameterTypes[0])) {
            return args[0];
        }

        Map<String, Object> bodyMap = new HashMap<>();
        for (int i = 0; i < apiParams.length; i++) {
            String parameterName = apiParams[i].getName();
            Object arg = args[i];
            Class<?> parameterType = parameterTypes[i];
            if (isSimpleType(parameterType)) {
                // 简单类型
                bodyMap.put(parameterName, arg);
            } else {
                Map<String, Object> map = beanToMap(arg);
                bodyMap.putAll(map);
            }
        }
        return bodyMap;
    }

    private Map<String, Object> beanToMap(Object arg) {
        if (arg == null) {
            return Collections.emptyMap();
        }
        try {
            return doBeanToMap(arg);
        } catch (IllegalAccessException e) {
            throw new OpenApiCreationException("bean转map错误," + JsonKit.toString(arg), e);
        }
    }

    private Map<String, Object> doBeanToMap(Object arg) throws IllegalAccessException {
        Map<String, Object> map = new HashMap<>();
        Field[] declaredFields = arg.getClass().getDeclaredFields();
        for (Field declaredField : declaredFields) {
            String fieldName = declaredField.getName();
            declaredField.setAccessible(true);
            Object value = declaredField.get(arg);
            map.put(fieldName, value);
        }
        return map;
    }

    private boolean isSimpleType(Class<?> klass) {
        if (ClassKit.isPrimitiveOrWrapper(klass) || klass == String.class || Map.class.isAssignableFrom(klass)
                || Collection.class.isAssignableFrom(klass)) {
            return true;
        }
        return false;
    }

    private ApiParamVO[] lookupParameterNames(Method method) {
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();

        ApiParamVO[] params = new ApiParamVO[parameterAnnotations.length];
        for (int i = 0; i < parameterAnnotations.length; i++) {
            Annotation[] parameterAnnotation = parameterAnnotations[i];
            for (Annotation annotation : parameterAnnotation) {
                if (annotation instanceof ApiParam) {
                    ApiParam apiParam = (ApiParam) annotation;
                    ApiParamVO apiParamVO = new ApiParamVO().setName(apiParam.value());
                    params[i] = apiParamVO;
                }
            }
        }
        return params;
    }

    private void initializePublic(Method method, OpenApiRequest request) {
        request.setHost(GatewayConfig.getAddress());
        request.setSignKey(this.openApiFactory.getSignKey());
    }

    private OpenApiVO getOpenApi(Method method) {

        OpenApi openApi = method.getAnnotation(OpenApi.class);
        if (openApi == null) {
            String url = getPath(method);

            OpenApiLevel level = OpenApiLevel.PUBLIC;
            return new OpenApiVO().setLevel(level).setPath(url);
        }
        String url = openApi.path();
        if (Strings.isNullOrEmpty(url)) {
            url = getPath(method);
        }
        OpenApiLevel level = openApi.level();
        return new OpenApiVO().setLevel(level).setPath(url);
    }

    private String getPath(Method method) {
        // 没有定义, 按照规则自动生成
        Class<?> declaringClass = method.getDeclaringClass();
        ApiResource apiResource = declaringClass.getAnnotation(ApiResource.class);
        String resource;
        if (apiResource != null) {
            resource = apiResource.value();
        } else {
            String api = declaringClass.getSimpleName().split("Api")[0];
            resource = api.substring(0, 1).toLowerCase() + api.substring(1);
        }
        return "/v1/" + this.serviceInfo.getService() + "/" + resource + "/" + method.getName();
    }
}
