package com.foresee.open.sdk.api;

import com.foresee.open.sdk.auth.InternalAccessTokenApi;
import com.foresee.open.sdk.auth.vo.AccessToken;
import com.foresee.open.sdk.client.OpenApiFactory;
import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.*;

import java.util.Date;
import java.util.Objects;
import java.util.concurrent.*;

/**
 * AccessTokenHelper
 *
 * @author chenwenlong@foresee.com.cn
 * @version 1.0
 */
public class AccessTokenHelper {

    // 这里应该和CacheGet一样自动刷新
/*    private static Cache<String, AccessToken> accessTokenCache = CacheBuilder.newBuilder()
            .maximumSize(10000)
            .expireAfterAccess(1, TimeUnit.DAYS)
            .build();*/

    private static ThreadFactory threadFactory = new ThreadFactoryBuilder()
            .setNameFormat("AccessToken-CacheRefresher-pool-%d")
//            .setUncaughtExceptionHandler((t,e) -> logger.error("guava cache refresh error", e))
            .setDaemon(true).build();

    private static ExecutorService parentExecutor = new ThreadPoolExecutor(Runtime.getRuntime().availableProcessors(),
            100,
            1,
            TimeUnit.MINUTES,
            new ArrayBlockingQueue<Runnable>(1000, false),
            threadFactory);

    final static ListeningExecutorService backgroundRefreshPools = MoreExecutors.listeningDecorator(parentExecutor);

    /**
     * 刷新token
     */
    private static LoadingCache<AccessTokenKey, AccessToken> accessTokenCache = CacheBuilder.newBuilder()
            .maximumSize(10000)
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .build(new CacheLoader<AccessTokenKey, AccessToken>() {
                @Override
                public AccessToken load(AccessTokenKey accessTokenKey) {
                    return generateValue(accessTokenKey);
                }

                private AccessToken generateValue(AccessTokenKey accessTokenKey) {
                    InternalAccessTokenApi internalAccessTokenApi = accessTokenKey.openApiFactory.create(InternalAccessTokenApi.class);
                    String userToken = accessTokenKey.getUserToken();
                    AccessToken accessToken;
                    if (Strings.isNullOrEmpty(userToken)) {
                        accessToken = internalAccessTokenApi
                                .getByApp(accessTokenKey.getProviderAppId());
                    } else {
                        accessToken = internalAccessTokenApi.getByUserToken(userToken, accessTokenKey.getProviderAppId());
                    }
                    Integer expireTimes = accessToken.getExpireTimes();
                    Date expireDate = new Date(System.currentTimeMillis() + expireTimes * 1000);
                    accessToken.expireDate(expireDate);
                    return accessToken;
                }

                @Override
                public ListenableFuture<AccessToken> reload(final AccessTokenKey key, AccessToken oldValue) throws Exception {
                    // 10分钟误差
                    int deviation = 10 * 60 * 1000;
                    if (oldValue.expireDate().getTime() - System.currentTimeMillis() <= deviation) {
                        // 如果旧值已经失效, 立即获取新的
                        return Futures.immediateFuture(load(key));
                    }
                    // 后台刷新
                    Callable<AccessToken> callable = new Callable<AccessToken>() {
                        @Override
                        public AccessToken call() {
                            return load(key);
                        }
                    };
                    return backgroundRefreshPools.submit(callable);
                }
            });


    private AccessTokenHelper() {
    }

    public static AccessToken getAppAccessToken(final OpenApiFactory openApiFactory, final String providerAppId) {
        AccessTokenKey accessTokenKey = new AccessTokenKey(openApiFactory)
                .setConsumerAppId(openApiFactory.getAppId())
                .setProviderAppId(providerAppId);

        return accessTokenCache.getUnchecked(accessTokenKey);
    }

    public static AccessToken getUserAccessToken(final OpenApiFactory openApiFactory,
                                                 final String providerAppId,
                                                 final String userToken) {
        AccessTokenKey accessTokenKey = new AccessTokenKey(openApiFactory)
                .setConsumerAppId(openApiFactory.getAppId())
                .setUserToken(userToken)
                .setProviderAppId(providerAppId);

        return accessTokenCache.getUnchecked(accessTokenKey);
    }

    public static void cleanCache() {
        accessTokenCache.invalidateAll();
    }


    static class AccessTokenKey {
        private OpenApiFactory openApiFactory;
        private String consumerAppId;
        private String providerAppId;
        private String userToken;

        public AccessTokenKey(OpenApiFactory openApiFactory) {
            this.openApiFactory = openApiFactory;
        }

        public String getConsumerAppId() {
            return consumerAppId;
        }

        public AccessTokenKey setConsumerAppId(String consumerAppId) {
            this.consumerAppId = consumerAppId;
            return this;
        }

        public String getProviderAppId() {
            return providerAppId;
        }

        public AccessTokenKey setProviderAppId(String providerAppId) {
            this.providerAppId = providerAppId;
            return this;
        }

        public String getUserToken() {
            return userToken;
        }

        public AccessTokenKey setUserToken(String userToken) {
            this.userToken = userToken;
            return this;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            AccessTokenKey that = (AccessTokenKey) o;
            return Objects.equals(consumerAppId, that.consumerAppId) &&
                    Objects.equals(providerAppId, that.providerAppId) &&
                    Objects.equals(userToken, that.userToken);
        }

        @Override
        public int hashCode() {
            return Objects.hash(consumerAppId, providerAppId, userToken);
        }
    }
}

